import { z } from 'zod';

// Phone validation schema
export const phoneSchema = z.object({
    phone: z
        .string()
        .min(1, 'מספר הפלאפון נדרש')
        .min(10, 'מספר הפלאפון חייב להכיל לפחות 10 תווים')
        .max(15, 'מספר הפלאפון לא יכול להכיל יותר מ-15 תווים')
        .regex(
            /^[0-9\-\(\)\s\+]+$/,
            'מספר פלאפון לא תקין'
        ),
    domain: z
        .string()
        .min(1, 'שם הדומיין נדרש')
        .min(3, 'שם הדומיין חייב להכיל לפחות 3 תווים')
        .max(255, 'שם הדומיין לא יכול להכיל יותר מ-255 תווים')
        .regex(
            /^[a-zA-Z0-9]([a-zA-Z0-9\-]{0,61}[a-zA-Z0-9])?(\.[a-zA-Z0-9]([a-zA-Z0-9\-]{0,61}[a-zA-Z0-9])?)*$/,
            'שם דומיין לא תקין'
        ),
    website: z
        .string()
        .min(1, 'כתובת האתר נדרשת')
        .url('כתובת אתר לא תקינה')
        .refine(
            (url) => url.startsWith('https://'),
            'כתובת האתר חייבת להתחיל ב-https://'
        ),
    owner: z
        .string()
        .min(1, 'שם הבעלים נדרש')
        .min(2, 'שם הבעלים חייב להכיל לפחות 2 תווים')
        .max(100, 'שם הבעלים לא יכול להכיל יותר מ-100 תווים'),
    email: z
        .string()
        .min(1, 'כתובת אימייל נדרשת')
        .email('כתובת אימייל לא תקינה')
        .max(255, 'כתובת אימייל לא יכולה להכיל יותר מ-255 תווים'),
    package: z
        .string()
        .min(1, 'סוג החבילה נדרש')
        .max(100, 'סוג החבילה לא יכול להכיל יותר מ-100 תווים'),
    amount: z
        .number()
        .min(0, 'הסכום חייב להיות חיובי')
        .max(999999.99, 'הסכום לא יכול להיות יותר מ-999,999.99'),
    status: z.enum(['paid', 'pending', 'overdue', 'free', 'paused', 'delete_backup', 'delete_no_backup', 'renewal']).refine(
        (val) => ['paid', 'pending', 'overdue', 'free', 'paused', 'delete_backup', 'delete_no_backup', 'renewal'].includes(val),
        { message: 'סטטוס לא תקין' }
    ),
    priority: z.enum(['high', 'medium', 'low', 'to-delete', 'to-check']).refine(
        (val) => ['high', 'medium', 'low', 'to-delete', 'to-check'].includes(val),
        { message: 'עדיפות לא תקינה' }
    ),
    startDate: z
        .string()
        .min(1, 'תאריך התחלה נדרש')
        .refine(
            (date) => {
                const dateObj = new Date(date);
                return !isNaN(dateObj.getTime());
            },
            'תאריך לא תקין'
        ),
    nextPayment: z
        .string()
        .min(1, 'תאריך התשלום הבא נדרש')
        .refine(
            (date) => {
                const dateObj = new Date(date);
                return !isNaN(dateObj.getTime());
            },
            'תאריך לא תקין'
        ),
    notes: z.string().optional(),
});

// Filter validation schema
export const filterSchema = z.object({
    search: z.string().optional(),
    priority: z.enum(['all', 'high', 'medium', 'low']).optional(),
    status: z.enum(['all', 'paid', 'pending', 'overdue']).optional(),
    showEntries: z.number().min(10).max(100).optional(),
});

// Bulk operation validation schema
export const bulkOperationSchema = z.object({
    phoneIds: z.array(z.string()).min(1, 'יש לבחור לפחות פלאפון אחד'),
    operation: z.enum(['delete', 'updateStatus', 'updatePriority']),
    newStatus: z.enum(['paid', 'pending', 'overdue']).optional(),
    newPriority: z.enum(['high', 'medium', 'low']).optional(),
});

// Export validation schema
export const exportSchema = z.object({
    format: z.enum(['pdf', 'excel', 'csv']),
    includeColumns: z.array(z.string()).min(1, 'יש לבחור לפחות עמודה אחת'),
    dateRange: z.object({
        from: z.string().optional(),
        to: z.string().optional(),
    }).optional(),
});

// Type exports
export type PhoneFormData = z.infer<typeof phoneSchema>;
export type FilterFormData = z.infer<typeof filterSchema>;
export type BulkOperationData = z.infer<typeof bulkOperationSchema>;
export type ExportData = z.infer<typeof exportSchema>;

// Validation helper functions
export const validatePhone = (data: unknown) => {
    return phoneSchema.safeParse(data);
};

export const validateFilters = (data: unknown) => {
    return filterSchema.safeParse(data);
};

export const validateBulkOperation = (data: unknown) => {
    return bulkOperationSchema.safeParse(data);
};

export const validateExport = (data: unknown) => {
    return exportSchema.safeParse(data);
};

// Error message helpers
export const getFieldError = (errors: Record<string, { message?: string }> | undefined, field: string): string | undefined => {
    return errors?.[field]?.message;
};

export const hasFieldError = (errors: Record<string, { message?: string }> | undefined, field: string): boolean => {
    return !!errors?.[field];
};

// Hebrew error messages
export const hebrewErrorMessages = {
    required: 'שדה זה נדרש',
    invalidEmail: 'כתובת אימייל לא תקינה',
    invalidUrl: 'כתובת אתר לא תקינה',
    invalidPhone: 'שם פלאפון לא תקין',
    minLength: (min: number) => `חייב להכיל לפחות ${min} תווים`,
    maxLength: (max: number) => `לא יכול להכיל יותר מ-${max} תווים`,
    minValue: (min: number) => `חייב להיות לפחות ${min}`,
    maxValue: (max: number) => `לא יכול להיות יותר מ-${max}`,
    futureDate: 'התאריך חייב להיות בעתיד',
    pastDate: 'התאריך חייב להיות בעבר',
};
