import { z } from 'zod';

// Customer validation schema
export const CustomerFormSchema = z.object({
    name: z.string().min(1, 'שם הלקוח נדרש'),
    email: z.string().email('כתובת אימייל לא תקינה').optional().or(z.literal('')),
    phone: z.string().optional(),
    company: z.string().optional(),
    status: z.enum(['active', 'inactive', 'lead'], {
        errorMap: () => ({ message: 'סטטוס לא תקין' })
    }),
    stage: z.enum(['quote', 'project', 'follow_up', 'closed'], {
        errorMap: () => ({ message: 'שלב לא תקין' })
    }),
    assigned_sales_rep: z.string().optional(),
    value: z.number().min(0, 'ערך חייב להיות חיובי'),
    notes: z.string().optional()
});

// Domain validation schema
export const DomainFormSchema = z.object({
    customer_id: z.string().min(1, 'לקוח נדרש'),
    domain_name: z.string().min(1, 'שם דומיין נדרש'),
    website: z.string().url('כתובת אתר לא תקינה').optional().or(z.literal('')),
    owner: z.string().optional(),
    email: z.string().email('כתובת אימייל לא תקינה').optional().or(z.literal('')),
    package: z.string().optional(),
    amount: z.number().min(0, 'סכום חייב להיות חיובי').optional(),
    status: z.enum(['active', 'inactive', 'pending', 'expired'], {
        errorMap: () => ({ message: 'סטטוס לא תקין' })
    }),
    priority: z.enum(['low', 'medium', 'high'], {
        errorMap: () => ({ message: 'עדיפות לא תקינה' })
    }),
    next_payment: z.string().optional(),
    notes: z.string().optional()
});

// Project validation schema
export const ProjectFormSchema = z.object({
    customer_id: z.string().min(1, 'לקוח נדרש'),
    title: z.string().min(1, 'כותרת הפרויקט נדרשת'),
    description: z.string().optional(),
    status: z.enum(['planning', 'in_progress', 'completed', 'on_hold', 'cancelled'], {
        errorMap: () => ({ message: 'סטטוס לא תקין' })
    }),
    start_date: z.string().optional(),
    end_date: z.string().optional(),
    budget: z.number().min(0, 'תקציב חייב להיות חיובי').optional(),
    progress: z.number().min(0).max(100, 'התקדמות חייבת להיות בין 0 ל-100'),
    assigned_team: z.string().optional(),
    notes: z.string().optional()
});

// Quote validation schema
export const QuoteFormSchema = z.object({
    customer_id: z.string().min(1, 'לקוח נדרש'),
    title: z.string().min(1, 'כותרת ההצעה נדרשת'),
    amount: z.number().min(0, 'סכום חייב להיות חיובי'),
    status: z.enum(['pending', 'sent', 'approved', 'rejected', 'expired'], {
        errorMap: () => ({ message: 'סטטוס לא תקין' })
    }),
    valid_until: z.string().optional(),
    description: z.string().optional(),
    terms: z.string().optional(),
    notes: z.string().optional()
});

// Payment validation schema
export const PaymentFormSchema = z.object({
    customer_id: z.string().min(1, 'לקוח נדרש'),
    domain_id: z.string().optional(),
    project_id: z.string().optional(),
    quote_id: z.string().optional(),
    amount: z.number().min(0, 'סכום חייב להיות חיובי'),
    payment_type: z.enum(['domain_renewal', 'project_payment', 'quote_payment', 'service_payment', 'other'], {
        errorMap: () => ({ message: 'סוג תשלום לא תקין' })
    }),
    status: z.enum(['pending', 'completed', 'failed', 'cancelled', 'refunded'], {
        errorMap: () => ({ message: 'סטטוס לא תקין' })
    }),
    payment_method: z.enum(['credit_card', 'bank_transfer', 'paypal', 'cash', 'check', 'other'], {
        errorMap: () => ({ message: 'אמצעי תשלום לא תקין' })
    }),
    payment_date: z.string().optional(),
    due_date: z.string().optional(),
    invoice_number: z.string().optional(),
    description: z.string().optional(),
    notes: z.string().optional()
});

// Activity validation schema
export const ActivityFormSchema = z.object({
    customer_id: z.string().min(1, 'לקוח נדרש'),
    activity_type: z.enum(['call', 'email', 'meeting', 'note', 'task', 'reminder', 'payment', 'other'], {
        errorMap: () => ({ message: 'סוג פעילות לא תקין' })
    }),
    title: z.string().min(1, 'כותרת הפעילות נדרשת'),
    description: z.string().optional(),
    status: z.enum(['pending', 'completed', 'cancelled'], {
        errorMap: () => ({ message: 'סטטוס לא תקין' })
    }),
    priority: z.enum(['low', 'medium', 'high'], {
        errorMap: () => ({ message: 'עדיפות לא תקינה' })
    }),
    due_date: z.string().optional(),
    assigned_to: z.string().optional(),
    duration_minutes: z.number().min(0, 'משך זמן חייב להיות חיובי').optional(),
    notes: z.string().optional()
});

// Reminder validation schema
export const ReminderFormSchema = z.object({
    customer_id: z.string().min(1, 'לקוח נדרש'),
    domain_id: z.string().optional(),
    project_id: z.string().optional(),
    reminder_type: z.enum(['domain_renewal', 'payment_due', 'project_deadline', 'follow_up', 'meeting', 'other'], {
        errorMap: () => ({ message: 'סוג תזכורת לא תקין' })
    }),
    title: z.string().min(1, 'כותרת התזכורת נדרשת'),
    description: z.string().optional(),
    reminder_date: z.string().min(1, 'תאריך תזכורת נדרש'),
    status: z.enum(['pending', 'completed', 'cancelled', 'overdue'], {
        errorMap: () => ({ message: 'סטטוס לא תקין' })
    }),
    priority: z.enum(['low', 'medium', 'high'], {
        errorMap: () => ({ message: 'עדיפות לא תקינה' })
    }),
    recurring: z.enum(['none', 'daily', 'weekly', 'monthly', 'yearly'], {
        errorMap: () => ({ message: 'חזרה לא תקינה' })
    }),
    notes: z.string().optional()
});

// Email Service validation schema
export const EmailServiceFormSchema = z.object({
    domain_id: z.string().min(1, 'דומיין נדרש'),
    service_type: z.enum(['hosting', 'email_accounts', 'ssl_certificate', 'backup', 'security', 'other'], {
        errorMap: () => ({ message: 'סוג שירות לא תקין' })
    }),
    service_name: z.string().min(1, 'שם השירות נדרש'),
    provider: z.string().optional(),
    status: z.enum(['active', 'inactive', 'pending', 'suspended'], {
        errorMap: () => ({ message: 'סטטוס לא תקין' })
    }),
    start_date: z.string().optional(),
    end_date: z.string().optional(),
    monthly_cost: z.number().min(0, 'עלות חודשית חייבת להיות חיובית').optional(),
    features: z.string().optional(),
    notes: z.string().optional()
});

// Technical Service validation schema
export const TechnicalServiceFormSchema = z.object({
    domain_id: z.string().min(1, 'דומיין נדרש'),
    service_type: z.enum(['dns_management', 'ssl_certificate', 'cdn', 'monitoring', 'backup', 'security', 'performance', 'other'], {
        errorMap: () => ({ message: 'סוג שירות לא תקין' })
    }),
    service_name: z.string().min(1, 'שם השירות נדרש'),
    provider: z.string().optional(),
    status: z.enum(['active', 'inactive', 'pending', 'maintenance'], {
        errorMap: () => ({ message: 'סטטוס לא תקין' })
    }),
    start_date: z.string().optional(),
    end_date: z.string().optional(),
    monthly_cost: z.number().min(0, 'עלות חודשית חייבת להיות חיובית').optional(),
    configuration: z.string().optional(),
    notes: z.string().optional()
});

// Type exports for forms
export type CustomerFormData = z.infer<typeof CustomerFormSchema>;
export type DomainFormData = z.infer<typeof DomainFormSchema>;
export type ProjectFormData = z.infer<typeof ProjectFormSchema>;
export type QuoteFormData = z.infer<typeof QuoteFormSchema>;
export type PaymentFormData = z.infer<typeof PaymentFormSchema>;
export type ActivityFormData = z.infer<typeof ActivityFormSchema>;
export type ReminderFormData = z.infer<typeof ReminderFormSchema>;
export type EmailServiceFormData = z.infer<typeof EmailServiceFormSchema>;
export type TechnicalServiceFormData = z.infer<typeof TechnicalServiceFormSchema>;
