// Comprehensive CRM Types - Complete system with relationships

export interface Customer {
    id: string;
    name: string;
    email?: string;
    phone?: string;
    company?: string;
    status: 'active' | 'inactive' | 'lead';
    stage: 'quote' | 'project' | 'follow_up' | 'closed';
    assigned_sales_rep?: string;
    value: number;
    notes?: string;
    created_date: string;
    last_activity: string;
    updated_at: string;
}

export interface Domain {
    id: string;
    customer_id: string;
    domain_name: string;
    website?: string;
    owner?: string;
    email?: string;
    package?: string;
    amount?: number;
    status: 'active' | 'inactive' | 'pending' | 'expired';
    priority: 'low' | 'medium' | 'high';
    next_payment?: string;
    last_update?: string;
    notes?: string;
    created_date: string;
    updated_at: string;
}

export interface Project {
    id: string;
    customer_id: string;
    title: string;
    description?: string;
    status: 'planning' | 'in_progress' | 'completed' | 'on_hold' | 'cancelled';
    start_date?: string;
    end_date?: string;
    budget?: number;
    progress: number;
    assigned_team?: string;
    notes?: string;
    created_date: string;
    updated_at: string;
}

export interface Quote {
    id: string;
    customer_id: string;
    title: string;
    amount: number;
    status: 'pending' | 'sent' | 'approved' | 'rejected' | 'expired';
    valid_until?: string;
    description?: string;
    terms?: string;
    notes?: string;
    created_date: string;
    updated_at: string;
}

export interface Payment {
    id: string;
    customer_id: string;
    domain_id?: string;
    project_id?: string;
    quote_id?: string;
    amount: number;
    payment_type: 'domain_renewal' | 'project_payment' | 'quote_payment' | 'service_payment' | 'other';
    status: 'pending' | 'completed' | 'failed' | 'cancelled' | 'refunded';
    payment_method: 'credit_card' | 'bank_transfer' | 'paypal' | 'cash' | 'check' | 'other';
    payment_date?: string;
    due_date?: string;
    invoice_number?: string;
    description?: string;
    notes?: string;
    created_date: string;
    updated_at: string;
}

export interface Activity {
    id: string;
    customer_id: string;
    activity_type: 'call' | 'email' | 'meeting' | 'note' | 'task' | 'reminder' | 'payment' | 'other';
    title: string;
    description?: string;
    status: 'pending' | 'completed' | 'cancelled';
    priority: 'low' | 'medium' | 'high';
    due_date?: string;
    completed_date?: string;
    assigned_to?: string;
    duration_minutes?: number;
    outcome?: string;
    notes?: string;
    created_date: string;
    updated_at: string;
}

export interface Reminder {
    id: string;
    customer_id: string;
    domain_id?: string;
    project_id?: string;
    reminder_type: 'domain_renewal' | 'payment_due' | 'project_deadline' | 'follow_up' | 'meeting' | 'other';
    title: string;
    description?: string;
    reminder_date: string;
    status: 'pending' | 'completed' | 'cancelled' | 'overdue';
    priority: 'low' | 'medium' | 'high';
    recurring: 'none' | 'daily' | 'weekly' | 'monthly' | 'yearly';
    completed_date?: string;
    notes?: string;
    created_date: string;
    updated_at: string;
}

export interface EmailService {
    id: string;
    domain_id: string;
    service_type: 'hosting' | 'email_accounts' | 'ssl_certificate' | 'backup' | 'security' | 'other';
    service_name: string;
    provider?: string;
    status: 'active' | 'inactive' | 'pending' | 'suspended';
    start_date?: string;
    end_date?: string;
    monthly_cost?: number;
    features?: string;
    notes?: string;
    created_date: string;
    updated_at: string;
}

export interface TechnicalService {
    id: string;
    domain_id: string;
    service_type: 'dns_management' | 'ssl_certificate' | 'cdn' | 'monitoring' | 'backup' | 'security' | 'performance' | 'other';
    service_name: string;
    provider?: string;
    status: 'active' | 'inactive' | 'pending' | 'maintenance';
    start_date?: string;
    end_date?: string;
    monthly_cost?: number;
    configuration?: string;
    notes?: string;
    created_date: string;
    updated_at: string;
}

// Form types for creating/editing entities
export interface CustomerForm {
    name: string;
    email?: string;
    phone?: string;
    company?: string;
    status: 'active' | 'inactive' | 'lead';
    stage: 'quote' | 'project' | 'follow_up' | 'closed';
    assigned_sales_rep?: string;
    value: number;
    notes?: string;
}

export interface DomainForm {
    customer_id: string;
    domain_name: string;
    website?: string;
    owner?: string;
    email?: string;
    package?: string;
    amount?: number;
    status: 'active' | 'inactive' | 'pending' | 'expired';
    priority: 'low' | 'medium' | 'high';
    next_payment?: string;
    notes?: string;
}

export interface ProjectForm {
    customer_id: string;
    title: string;
    description?: string;
    status: 'planning' | 'in_progress' | 'completed' | 'on_hold' | 'cancelled';
    start_date?: string;
    end_date?: string;
    budget?: number;
    progress: number;
    assigned_team?: string;
    notes?: string;
}

export interface QuoteForm {
    customer_id: string;
    title: string;
    amount: number;
    status: 'pending' | 'sent' | 'approved' | 'rejected' | 'expired';
    valid_until?: string;
    description?: string;
    terms?: string;
    notes?: string;
}

export interface PaymentForm {
    customer_id: string;
    domain_id?: string;
    project_id?: string;
    quote_id?: string;
    amount: number;
    payment_type: 'domain_renewal' | 'project_payment' | 'quote_payment' | 'service_payment' | 'other';
    status: 'pending' | 'completed' | 'failed' | 'cancelled' | 'refunded';
    payment_method: 'credit_card' | 'bank_transfer' | 'paypal' | 'cash' | 'check' | 'other';
    payment_date?: string;
    due_date?: string;
    invoice_number?: string;
    description?: string;
    notes?: string;
}

export interface ActivityForm {
    customer_id: string;
    activity_type: 'call' | 'email' | 'meeting' | 'note' | 'task' | 'reminder' | 'payment' | 'other';
    title: string;
    description?: string;
    status: 'pending' | 'completed' | 'cancelled';
    priority: 'low' | 'medium' | 'high';
    due_date?: string;
    assigned_to?: string;
    duration_minutes?: number;
    notes?: string;
}

export interface ReminderForm {
    customer_id: string;
    domain_id?: string;
    project_id?: string;
    reminder_type: 'domain_renewal' | 'payment_due' | 'project_deadline' | 'follow_up' | 'meeting' | 'other';
    title: string;
    description?: string;
    reminder_date: string;
    status: 'pending' | 'completed' | 'cancelled' | 'overdue';
    priority: 'low' | 'medium' | 'high';
    recurring: 'none' | 'daily' | 'weekly' | 'monthly' | 'yearly';
    notes?: string;
}

export interface EmailServiceForm {
    domain_id: string;
    service_type: 'hosting' | 'email_accounts' | 'ssl_certificate' | 'backup' | 'security' | 'other';
    service_name: string;
    provider?: string;
    status: 'active' | 'inactive' | 'pending' | 'suspended';
    start_date?: string;
    end_date?: string;
    monthly_cost?: number;
    features?: string;
    notes?: string;
}

export interface TechnicalServiceForm {
    domain_id: string;
    service_type: 'dns_management' | 'ssl_certificate' | 'cdn' | 'monitoring' | 'backup' | 'security' | 'performance' | 'other';
    service_name: string;
    provider?: string;
    status: 'active' | 'inactive' | 'pending' | 'maintenance';
    start_date?: string;
    end_date?: string;
    monthly_cost?: number;
    configuration?: string;
    notes?: string;
}

// Dashboard statistics interface
export interface ComprehensiveStats {
    // Customer stats
    totalCustomers: number;
    activeCustomers: number;
    leadCustomers: number;
    inactiveCustomers: number;

    // Domain stats
    totalDomains: number;
    activeDomains: number;
    expiringDomains: number;
    expiredDomains: number;

    // Project stats
    totalProjects: number;
    activeProjects: number;
    completedProjects: number;
    overdueProjects: number;

    // Quote stats
    totalQuotes: number;
    pendingQuotes: number;
    approvedQuotes: number;
    rejectedQuotes: number;

    // Payment stats
    totalPayments: number;
    pendingPayments: number;
    completedPayments: number;
    overduePayments: number;
    totalRevenue: number;

    // Activity stats
    totalActivities: number;
    pendingActivities: number;
    completedActivities: number;
    overdueActivities: number;

    // Reminder stats
    totalReminders: number;
    pendingReminders: number;
    overdueReminders: number;
    completedReminders: number;

    // Service stats
    totalEmailServices: number;
    activeEmailServices: number;
    totalTechnicalServices: number;
    activeTechnicalServices: number;
}

// Relationship types for complex queries
export interface CustomerWithRelations extends Customer {
    domains: Domain[];
    projects: Project[];
    quotes: Quote[];
    payments: Payment[];
    activities: Activity[];
    reminders: Reminder[];
}

export interface DomainWithRelations extends Domain {
    customer: Customer;
    email_services: EmailService[];
    technical_services: TechnicalService[];
    payments: Payment[];
    reminders: Reminder[];
}

export interface ProjectWithRelations extends Project {
    customer: Customer;
    payments: Payment[];
    reminders: Reminder[];
}

export interface QuoteWithRelations extends Quote {
    customer: Customer;
    payments: Payment[];
}
