import { create } from 'zustand';
import { persist } from 'zustand/middleware';

interface UIState {
    // Modal states
    isAddPhoneModalOpen: boolean;
    isEditPhoneModalOpen: boolean;
    isDeleteConfirmModalOpen: boolean;
    isSettingsModalOpen: boolean;
    isImportCSVModalOpen: boolean;
    isExportCSVModalOpen: boolean;

    // Selected phone for editing
    selectedPhoneId: string | null;

    // Sidebar state
    isSidebarOpen: boolean;

    // Theme
    isDarkMode: boolean;

    // Loading states
    isGlobalLoading: boolean;

    // Notifications
    notifications: Notification[];
}

interface UIActions {
    // Modal actions
    openAddPhoneModal: () => void;
    closeAddPhoneModal: () => void;
    openEditPhoneModal: (phoneId: string) => void;
    closeEditPhoneModal: () => void;
    openDeleteConfirmModal: (phoneId: string) => void;
    closeDeleteConfirmModal: () => void;
    openSettingsModal: () => void;
    closeSettingsModal: () => void;
    openImportCSVModal: () => void;
    closeImportCSVModal: () => void;
    openExportCSVModal: () => void;
    closeExportCSVModal: () => void;

    // Sidebar actions
    toggleSidebar: () => void;
    setSidebarOpen: (open: boolean) => void;

    // Theme actions
    toggleDarkMode: () => void;
    setDarkMode: (dark: boolean) => void;

    // Loading actions
    setGlobalLoading: (loading: boolean) => void;

    // Notification actions
    addNotification: (notification: Omit<Notification, 'id'>) => void;
    removeNotification: (id: string) => void;
    clearNotifications: () => void;
}

interface Notification {
    id: string;
    type: 'success' | 'error' | 'warning' | 'info';
    title: string;
    message: string;
    duration?: number;
    timestamp: Date;
}

export const useUIStore = create<UIState & UIActions>()(
    persist(
        (set, get) => ({
            // Initial state
            isAddPhoneModalOpen: false,
            isEditPhoneModalOpen: false,
            isDeleteConfirmModalOpen: false,
            isSettingsModalOpen: false,
            isImportCSVModalOpen: false,
            isExportCSVModalOpen: false,
            selectedPhoneId: null,
            isSidebarOpen: true,
            isDarkMode: false,
            isGlobalLoading: false,
            notifications: [],

            // Modal actions
            openAddPhoneModal: () => {
                set({ isAddPhoneModalOpen: true });
            },

            closeAddPhoneModal: () => {
                set({ isAddPhoneModalOpen: false });
            },

            openEditPhoneModal: (phoneId) => {
                set({
                    isEditPhoneModalOpen: true,
                    selectedPhoneId: phoneId,
                });
            },

            closeEditPhoneModal: () => {
                set({
                    isEditPhoneModalOpen: false,
                    selectedPhoneId: null,
                });
            },

            openDeleteConfirmModal: (phoneId) => {
                set({
                    isDeleteConfirmModalOpen: true,
                    selectedPhoneId: phoneId,
                });
            },

            closeDeleteConfirmModal: () => {
                set({
                    isDeleteConfirmModalOpen: false,
                    selectedPhoneId: null,
                });
            },

            openSettingsModal: () => {
                set({ isSettingsModalOpen: true });
            },

            closeSettingsModal: () => {
                set({ isSettingsModalOpen: false });
            },

            openImportCSVModal: () => {
                set({ isImportCSVModalOpen: true });
            },

            closeImportCSVModal: () => {
                set({ isImportCSVModalOpen: false });
            },

            openExportCSVModal: () => {
                set({ isExportCSVModalOpen: true });
            },

            closeExportCSVModal: () => {
                set({ isExportCSVModalOpen: false });
            },

            // Sidebar actions
            toggleSidebar: () => {
                set((state) => ({ isSidebarOpen: !state.isSidebarOpen }));
            },

            setSidebarOpen: (open) => {
                set({ isSidebarOpen: open });
            },

            // Theme actions
            toggleDarkMode: () => {
                set((state) => {
                    const newDarkMode = !state.isDarkMode;
                    // Save to localStorage
                    if (typeof window !== 'undefined') {
                        localStorage.setItem('darkMode', newDarkMode.toString());
                    }
                    return { isDarkMode: newDarkMode };
                });
            },

            setDarkMode: (dark) => {
                set({ isDarkMode: dark });
                if (typeof window !== 'undefined') {
                    localStorage.setItem('darkMode', dark.toString());
                }
            },

            // Loading actions
            setGlobalLoading: (loading) => {
                set({ isGlobalLoading: loading });
            },

            // Notification actions
            addNotification: (notification) => {
                const newNotification: Notification = {
                    ...notification,
                    id: Date.now().toString(),
                    timestamp: new Date(),
                };

                set((state) => ({
                    notifications: [...state.notifications, newNotification],
                }));

                // Auto remove after duration
                if (notification.duration) {
                    setTimeout(() => {
                        get().removeNotification(newNotification.id);
                    }, notification.duration);
                }
            },

            removeNotification: (id) => {
                set((state) => ({
                    notifications: state.notifications.filter((notification) => notification.id !== id),
                }));
            },

            clearNotifications: () => {
                set({ notifications: [] });
            },
        }),
        {
            name: 'ui-store',
            partialize: (state) => ({
                isSidebarOpen: state.isSidebarOpen,
                isDarkMode: state.isDarkMode,
                // Don't persist modal states - they should always start closed
            }),
        }
    )
);
