import { create } from 'zustand';
import { Phone, FilterOptions } from '@/types/phone';

interface PhoneState {
    phones: Phone[];
    filteredPhones: Phone[];
    filters: FilterOptions;
    selectedPhones: string[];
    isLoading: boolean;
    error: string | null;
}

interface PhoneActions {
    // Phone CRUD operations
    addPhone: (phone: Omit<Phone, 'id'>) => Promise<void>;
    updatePhone: (id: string, updates: Partial<Phone>) => Promise<void>;
    deletePhone: (id: string) => Promise<void>;
    deleteSelectedPhones: () => Promise<void>;
    fetchPhones: () => Promise<void>;

    // Selection management
    selectPhone: (id: string) => void;
    deselectPhone: (id: string) => void;
    selectAllPhones: () => void;
    clearSelection: () => void;
    togglePhoneSelection: (id: string) => void;

    // Filtering and search
    setFilters: (filters: Partial<FilterOptions>) => void;
    resetFilters: () => void;
    applyFilters: () => void;

    // Search
    searchPhones: (query: string) => void;

    // Sorting
    sortPhones: (field: keyof Phone, direction: 'asc' | 'desc') => void;

    // Loading and error states
    setLoading: (loading: boolean) => void;
    setError: (error: string | null) => void;

    // Bulk operations
    bulkUpdateStatus: (ids: string[], status: Phone['status']) => Promise<void>;
    bulkUpdatePriority: (ids: string[], priority: Phone['priority']) => Promise<void>;
    clearPhones: () => void;
}

const initialFilters: FilterOptions = {
    search: '',
    priority: 'all',
    status: 'all',
    showEntries: 25,
    nextPaymentFrom: undefined,
    nextPaymentTo: undefined,
};

// API helper functions
const apiCall = async (url: string, options: RequestInit = {}) => {
    console.log('🌐 API Call:', url, options);
    const response = await fetch(url, {
        headers: {
            'Content-Type': 'application/json',
            ...options.headers,
        },
        ...options,
    });

    console.log('📡 Response status:', response.status, response.statusText);

    if (!response.ok) {
        throw new Error(`API Error: ${response.status} ${response.statusText}`);
    }

    const data = await response.json();
    console.log('📦 Response data:', data);
    return data;
};

export const usePhoneStore = create<PhoneState & PhoneActions>()((set, get) => ({
    // Initial state
    phones: [],
    filteredPhones: [],
    filters: initialFilters,
    selectedPhones: [],
    isLoading: false,
    error: null,

    // Phone CRUD operations
    addPhone: async (phone) => {
        try {
            set({ isLoading: true, error: null });
            const newPhone = await apiCall('/api/phones', {
                method: 'POST',
                body: JSON.stringify(phone),
            });

            set((state) => ({
                phones: [...state.phones, newPhone],
            }));
            get().applyFilters();
        } catch (error) {
            console.error('Error adding phone:', error);
            set({ error: error instanceof Error ? error.message : 'Failed to add phone' });
        } finally {
            set({ isLoading: false });
        }
    },

    updatePhone: async (id, updates) => {
        try {
            set({ isLoading: true, error: null });
            await apiCall(`/api/phones/${id}`, {
                method: 'PUT',
                body: JSON.stringify(updates),
            });

            set((state) => ({
                phones: state.phones.map((phone) =>
                    phone.id === id ? { ...phone, ...updates, updatedAt: new Date().toISOString() } : phone
                ),
            }));
            get().applyFilters();
        } catch (error) {
            console.error('Error updating phone:', error);
            set({ error: error instanceof Error ? error.message : 'Failed to update phone' });
        } finally {
            set({ isLoading: false });
        }
    },

    deletePhone: async (id) => {
        try {
            set({ isLoading: true, error: null });
            await apiCall(`/api/phones/${id}`, {
                method: 'DELETE',
            });

            set((state) => ({
                phones: state.phones.filter((phone) => phone.id !== id),
                selectedPhones: state.selectedPhones.filter((phoneId) => phoneId !== id),
            }));
            get().applyFilters();
        } catch (error) {
            console.error('Error deleting phone:', error);
            set({ error: error instanceof Error ? error.message : 'Failed to delete phone' });
        } finally {
            set({ isLoading: false });
        }
    },

    deleteSelectedPhones: async () => {
        const { selectedPhones } = get();
        try {
            set({ isLoading: true, error: null });

            // Delete all selected phones
            await Promise.all(
                selectedPhones.map(id =>
                    apiCall(`/api/phones/${id}`, { method: 'DELETE' })
                )
            );

            set((state) => ({
                phones: state.phones.filter((phone) => !selectedPhones.includes(phone.id)),
                selectedPhones: [],
            }));
            get().applyFilters();
        } catch (error) {
            console.error('Error deleting selected phones:', error);
            set({ error: error instanceof Error ? error.message : 'Failed to delete selected phones' });
        } finally {
            set({ isLoading: false });
        }
    },

    fetchPhones: async () => {
        try {
            console.log('🔄 Fetching domains from comprehensive.db...');
            set({ isLoading: true, error: null });
            const response = await apiCall('/api/domains');
            const domains = response.domains || [];

            // Convert domains to phone format for compatibility
            const phones = domains.map((domain: any) => ({
                id: domain.id,
                phone: '', // Not used for domains
                domain: domain.domain_name,
                website: domain.domain_name,
                owner: domain.customer_name || domain.customer_id, // Use customer name if available
                email: domain.customer_email || '', // Use customer email if available
                package: 'Linux Mini Professionals',
                amount: domain.price || 1000,
                status: domain.status,
                priority: 'high',
                nextPayment: domain.expiry_date,
                startDate: domain.registration_date || domain.created_at || new Date().toISOString(),
                createdAt: domain.created_at || new Date().toISOString(),
                updatedAt: domain.updated_at || new Date().toISOString(),
                notes: domain.notes || ''
            }));

            console.log('✅ Domains fetched and converted:', phones.length, 'domains');
            set({ phones });
            get().applyFilters();
            console.log('✅ Filters applied');
        } catch (error) {
            console.error('❌ Error fetching domains:', error);
            set({ error: error instanceof Error ? error.message : 'Failed to fetch domains' });
        } finally {
            set({ isLoading: false });
        }
    },

    // Selection management
    selectPhone: (id) => {
        set((state) => ({
            selectedPhones: [...state.selectedPhones, id],
        }));
    },

    deselectPhone: (id) => {
        set((state) => ({
            selectedPhones: state.selectedPhones.filter((phoneId) => phoneId !== id),
        }));
    },

    selectAllPhones: () => {
        const { filteredPhones } = get();
        set({
            selectedPhones: filteredPhones.map((phone) => phone.id),
        });
    },

    clearSelection: () => {
        set({ selectedPhones: [] });
    },

    togglePhoneSelection: (id) => {
        const { selectedPhones } = get();
        if (selectedPhones.includes(id)) {
            get().deselectPhone(id);
        } else {
            get().selectPhone(id);
        }
    },

    // Filtering and search
    setFilters: (newFilters) => {
        set((state) => ({
            filters: { ...state.filters, ...newFilters },
        }));
        get().applyFilters();
    },

    resetFilters: () => {
        set({ filters: initialFilters });
        get().applyFilters();
    },

    applyFilters: () => {
        const { phones, filters } = get();

        let filtered = phones;

        // Search filter
        if (filters.search) {
            const searchTerm = filters.search.toLowerCase();
            filtered = filtered.filter((phone) =>
                phone.phone.toLowerCase().includes(searchTerm) ||
                phone.owner.toLowerCase().includes(searchTerm) ||
                phone.email.toLowerCase().includes(searchTerm) ||
                phone.package.toLowerCase().includes(searchTerm)
            );
        }

        // Priority filter
        if (filters.priority !== 'all') {
            filtered = filtered.filter((phone) => phone.priority === filters.priority);
        }

        // Status filter
        if (filters.status !== 'all') {
            filtered = filtered.filter((phone) => phone.status === filters.status);
        }

        // Renewal date range (nextPayment)
        if (filters.nextPaymentFrom) {
            const from = new Date(filters.nextPaymentFrom);
            filtered = filtered.filter((p) => new Date(p.nextPayment) >= from);
        }
        if (filters.nextPaymentTo) {
            const to = new Date(filters.nextPaymentTo);
            // include the whole day for 'to'
            to.setHours(23, 59, 59, 999);
            filtered = filtered.filter((p) => new Date(p.nextPayment) <= to);
        }

        set({ filteredPhones: filtered });
    },

    searchPhones: (query) => {
        get().setFilters({ search: query });
    },

    // Sorting
    sortPhones: (field, direction) => {
        const { filteredPhones } = get();
        const sorted = [...filteredPhones].sort((a, b) => {
            let aValue = a[field];
            let bValue = b[field];

            if (typeof aValue === 'string' && typeof bValue === 'string') {
                aValue = aValue.toLowerCase();
                bValue = bValue.toLowerCase();
            }

            if (direction === 'asc') {
                return aValue < bValue ? -1 : aValue > bValue ? 1 : 0;
            } else {
                return aValue > bValue ? -1 : aValue < bValue ? 1 : 0;
            }
        });

        set({ filteredPhones: sorted });
    },

    // Loading and error states
    setLoading: (loading) => {
        set({ isLoading: loading });
    },

    setError: (error) => {
        set({ error });
    },

    // Bulk operations
    bulkUpdateStatus: async (ids, status) => {
        try {
            set({ isLoading: true, error: null });

            // Update all selected phones
            await Promise.all(
                ids.map(id =>
                    apiCall(`/api/phones/${id}`, {
                        method: 'PUT',
                        body: JSON.stringify({ status }),
                    })
                )
            );

            set((state) => ({
                phones: state.phones.map((phone) =>
                    ids.includes(phone.id) ? { ...phone, status, updatedAt: new Date().toISOString() } : phone
                ),
                filteredPhones: state.filteredPhones.map((phone) =>
                    ids.includes(phone.id) ? { ...phone, status, updatedAt: new Date().toISOString() } : phone
                ),
            }));
        } catch (error) {
            console.error('Error updating status:', error);
            set({ error: error instanceof Error ? error.message : 'Failed to update status' });
        } finally {
            set({ isLoading: false });
        }
    },

    bulkUpdatePriority: async (ids, priority) => {
        try {
            set({ isLoading: true, error: null });

            // Update all selected phones
            await Promise.all(
                ids.map(id =>
                    apiCall(`/api/phones/${id}`, {
                        method: 'PUT',
                        body: JSON.stringify({ priority }),
                    })
                )
            );

            set((state) => ({
                phones: state.phones.map((phone) =>
                    ids.includes(phone.id) ? { ...phone, priority, updatedAt: new Date().toISOString() } : phone
                ),
                filteredPhones: state.filteredPhones.map((phone) =>
                    ids.includes(phone.id) ? { ...phone, priority, updatedAt: new Date().toISOString() } : phone
                ),
            }));
        } catch (error) {
            console.error('Error updating priority:', error);
            set({ error: error instanceof Error ? error.message : 'Failed to update priority' });
        } finally {
            set({ isLoading: false });
        }
    },

    clearPhones: () => {
        set({ phones: [], filteredPhones: [] });
    },
}));
