/**
 * Notes UI Store - Zustand
 * ניהול מצב UI של מודול הפתקים
 */

import { create } from 'zustand';
import { persist } from 'zustand/middleware';
import { FilterType, SortType } from '@/types/notes';

interface NotesUIStore {
    filter: FilterType;
    sortBy: SortType;
    searchQuery: string;
    isNoteEditorOpen: boolean;
    editingNoteId: string | null;
    isMobileSidebarOpen: boolean;

    // Actions
    setFilter: (filter: FilterType) => void;
    setSortBy: (sortBy: SortType) => void;
    setSearchQuery: (query: string) => void;
    openNoteEditor: (noteId?: string) => void;
    closeNoteEditor: () => void;
    toggleMobileSidebar: () => void;
    setMobileSidebar: (open: boolean) => void;
    resetFilters: () => void;
}

export const useNotesUIStore = create<NotesUIStore>()(
    persist(
        (set) => ({
            filter: 'all',
            sortBy: 'date',
            searchQuery: '',
            isNoteEditorOpen: false,
            editingNoteId: null,
            isMobileSidebarOpen: false,

            setFilter: (filter) => set({ filter }),

            setSortBy: (sortBy) => set({ sortBy }),

            setSearchQuery: (query) => set({ searchQuery: query }),

            openNoteEditor: (noteId) => set({
                isNoteEditorOpen: true,
                editingNoteId: noteId || null
            }),

            closeNoteEditor: () => set({
                isNoteEditorOpen: false,
                editingNoteId: null
            }),

            toggleMobileSidebar: () => set((state) => ({
                isMobileSidebarOpen: !state.isMobileSidebarOpen
            })),

            setMobileSidebar: (open) => set({ isMobileSidebarOpen: open }),

            resetFilters: () => set({
                filter: 'all',
                sortBy: 'date',
                searchQuery: ''
            }),
        }),
        {
            name: 'notes-ui-storage',
            partialize: (state) => ({
                filter: state.filter,
                sortBy: state.sortBy,
            }),
        }
    )
);



