/**
 * Notes Store - Zustand
 * ניהול מצב פתקים
 */

import { create } from 'zustand';
import { Note } from '@/types/notes';
import { searchNotes, sortNotes, filterNotes } from '@/src/lib/notes-helpers';

interface NotesStore {
    notes: Note[];
    isLoading: boolean;

    // Actions
    setNotes: (notes: Note[]) => void;
    addNote: (note: Note) => void;
    updateNote: (id: string, updates: Partial<Note>) => void;
    deleteNote: (id: string) => void;
    toggleDone: (id: string) => void;
    toggleImportant: (id: string) => void;
    toggleFlagged: (id: string) => void;
    duplicateNote: (id: string) => void;
    setLoading: (loading: boolean) => void;

    // Selectors
    getNotesByList: (listId: string | null) => Note[];
    getImportantNotes: () => Note[];
    getPlannedNotes: () => Note[];
    getCompletedNotes: () => Note[];
    getFlaggedNotes: () => Note[];
    getTodayNotes: () => Note[];
    getAllNotes: () => Note[];
    searchNotesQuery: (query: string) => Note[];

    // API Actions
    fetchNotes: (listId?: string, filter?: string, search?: string) => Promise<void>;
    createNote: (data: Partial<Note>) => Promise<void>;
    updateNoteApi: (id: string, updates: Partial<Note>) => Promise<void>;
    deleteNoteApi: (id: string) => Promise<void>;
}

export const useNotesStore = create<NotesStore>((set, get) => ({
    notes: [],
    isLoading: false,

    setNotes: (notes) => set({ notes }),

    addNote: (note) => set((state) => ({
        notes: [note, ...state.notes]
    })),

    updateNote: (id, updates) => set((state) => ({
        notes: state.notes.map(note =>
            note.id === id ? { ...note, ...updates, updatedAt: new Date().toISOString() } : note
        )
    })),

    deleteNote: (id) => set((state) => ({
        notes: state.notes.filter(note => note.id !== id)
    })),

    toggleDone: (id) => set((state) => ({
        notes: state.notes.map(note =>
            note.id === id ? { ...note, done: !note.done, updatedAt: new Date().toISOString() } : note
        )
    })),

    toggleImportant: (id) => set((state) => ({
        notes: state.notes.map(note =>
            note.id === id ? { ...note, important: !note.important, updatedAt: new Date().toISOString() } : note
        )
    })),

    toggleFlagged: (id) => set((state) => ({
        notes: state.notes.map(note =>
            note.id === id ? { ...note, flagged: !note.flagged, updatedAt: new Date().toISOString() } : note
        )
    })),

    duplicateNote: (id) => {
        const note = get().notes.find(n => n.id === id);
        if (note) {
            const duplicated: Note = {
                ...note,
                id: Date.now().toString() + Math.random().toString(36).substr(2, 9),
                title: `${note.title} (עותק)`,
                createdAt: new Date().toISOString(),
                updatedAt: new Date().toISOString(),
            };
            get().addNote(duplicated);
        }
    },

    setLoading: (loading) => set({ isLoading: loading }),

    // Selectors
    getNotesByList: (listId) => {
        const { notes } = get();
        if (listId === 'all' || listId === null) return notes;
        return notes.filter(n => n.listId === listId);
    },

    getImportantNotes: () => {
        const { notes } = get();
        return notes.filter(n => n.important && !n.done);
    },

    getPlannedNotes: () => {
        const { notes } = get();
        return notes.filter(n => n.dueDate && !n.done);
    },

    getCompletedNotes: () => {
        const { notes } = get();
        return notes.filter(n => n.done);
    },

    getFlaggedNotes: () => {
        const { notes } = get();
        return notes.filter(n => n.flagged);
    },

    getTodayNotes: () => {
        const { notes } = get();
        const today = new Date().toISOString().split('T')[0]; // YYYY-MM-DD
        return notes.filter(n => {
            if (n.done) return false;
            return (n.reminderDate === today) || (n.dueDate === today);
        });
    },

    getAllNotes: () => {
        return get().notes;
    },

    searchNotesQuery: (query) => {
        return searchNotes(get().notes, query);
    },

    // API Actions
    fetchNotes: async (listId, filter, search) => {
        set({ isLoading: true });
        try {
            const params = new URLSearchParams();
            if (listId) params.append('listId', listId);
            if (filter) params.append('filter', filter);
            if (search) params.append('search', search);

            const res = await fetch(`/api/notes?${params.toString()}`);
            const data = await res.json();
            set({ notes: data.notes || [], isLoading: false });
        } catch (error) {
            console.error('Error fetching notes:', error);
            set({ isLoading: false });
        }
    },

    createNote: async (data) => {
        try {
            const res = await fetch('/api/notes', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(data),
            });
            const result = await res.json();
            get().addNote(result.note);
        } catch (error) {
            console.error('Error creating note:', error);
        }
    },

    updateNoteApi: async (id, updates) => {
        try {
            const res = await fetch(`/api/notes/${id}`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(updates),
            });
            const result = await res.json();
            get().updateNote(id, result.note);
        } catch (error) {
            console.error('Error updating note:', error);
        }
    },

    deleteNoteApi: async (id) => {
        try {
            await fetch(`/api/notes/${id}`, { method: 'DELETE' });
            get().deleteNote(id);
        } catch (error) {
            console.error('Error deleting note:', error);
        }
    },
}));

