/**
 * Lists Store - Zustand
 * ניהול מצב רשימות
 */

import { create } from 'zustand';
import { persist } from 'zustand/middleware';
import { List } from '@/types/notes';

interface ListsStore {
    lists: List[];
    selectedListId: string | null;
    isLoading: boolean;

    // Actions
    setLists: (lists: List[]) => void;
    addList: (list: List) => void;
    updateList: (id: string, updates: Partial<List>) => void;
    deleteList: (id: string) => void;
    selectList: (id: string | null) => void;
    reorderLists: (fromIndex: number, toIndex: number) => void;
    setLoading: (loading: boolean) => void;

    // API Actions
    fetchLists: () => Promise<void>;
    createList: (data: { name: string; color?: string; icon?: string; sortOrder?: number }) => Promise<void>;
    updateListApi: (id: string, updates: Partial<List>) => Promise<void>;
    deleteListApi: (id: string, moveNotesToListId?: string) => Promise<void>;
}

export const useListsStore = create<ListsStore>()(
    persist(
        (set, get) => ({
            lists: [],
            selectedListId: null,
            isLoading: false,

            setLists: (lists) => set({ lists }),

            addList: (list) => set((state) => ({
                lists: [...state.lists, list].sort((a, b) => a.sortOrder - b.sortOrder)
            })),

            updateList: (id, updates) => set((state) => ({
                lists: state.lists.map(list =>
                    list.id === id ? { ...list, ...updates, updatedAt: new Date().toISOString() } : list
                )
            })),

            deleteList: (id) => set((state) => ({
                lists: state.lists.filter(list => list.id !== id),
                selectedListId: state.selectedListId === id ? null : state.selectedListId
            })),

            selectList: (id) => set({ selectedListId: id }),

            reorderLists: (fromIndex, toIndex) => set((state) => {
                const newLists = [...state.lists];
                const [moved] = newLists.splice(fromIndex, 1);
                newLists.splice(toIndex, 0, moved);

                return {
                    lists: newLists.map((list, index) => ({
                        ...list,
                        sortOrder: index
                    }))
                };
            }),

            setLoading: (loading) => set({ isLoading: loading }),

            // API Actions
            fetchLists: async () => {
                set({ isLoading: true });
                try {
                    const res = await fetch('/api/notes/lists');
                    const data = await res.json();
                    set({ lists: data.lists || [], isLoading: false });
                } catch (error) {
                    console.error('Error fetching lists:', error);
                    set({ isLoading: false });
                }
            },

            createList: async (data) => {
                try {
                    const res = await fetch('/api/notes/lists', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify(data),
                    });
                    const result = await res.json();
                    get().addList(result.list);
                } catch (error) {
                    console.error('Error creating list:', error);
                }
            },

            updateListApi: async (id, updates) => {
                try {
                    const res = await fetch(`/api/notes/lists/${id}`, {
                        method: 'PUT',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify(updates),
                    });
                    const result = await res.json();
                    get().updateList(id, result.list);
                } catch (error) {
                    console.error('Error updating list:', error);
                }
            },

            deleteListApi: async (id, moveNotesToListId) => {
                try {
                    const url = moveNotesToListId
                        ? `/api/notes/lists/${id}?moveNotesToListId=${moveNotesToListId}`
                        : `/api/notes/lists/${id}`;

                    await fetch(url, { method: 'DELETE' });
                    get().deleteList(id);
                } catch (error) {
                    console.error('Error deleting list:', error);
                }
            },
        }),
        {
            name: 'notes-lists-storage',
            partialize: (state) => ({
                selectedListId: state.selectedListId
            }),
        }
    )
);

