import { create } from 'zustand';
import { persist } from 'zustand/middleware';
import { Phone, FilterOptions } from '@/types/phone';

interface PhoneState {
    phones: Phone[];
    filteredPhones: Phone[];
    filters: FilterOptions;
    selectedPhones: string[];
    isLoading: boolean;
    error: string | null;
}

interface PhoneActions {
    // Phone CRUD operations
    addPhone: (phone: Omit<Phone, 'id'>) => void;
    updatePhone: (id: string, updates: Partial<Phone>) => void;
    deletePhone: (id: string) => void;
    deleteSelectedPhones: () => void;

    // Selection management
    selectPhone: (id: string) => void;
    deselectPhone: (id: string) => void;
    selectAllPhones: () => void;
    clearSelection: () => void;
    togglePhoneSelection: (id: string) => void;

    // Filtering and search
    setFilters: (filters: Partial<FilterOptions>) => void;
    resetFilters: () => void;
    applyFilters: () => void;

    // Search
    searchPhones: (query: string) => void;

    // Sorting
    sortPhones: (field: keyof Phone, direction: 'asc' | 'desc') => void;

    // Loading and error states
    setLoading: (loading: boolean) => void;
    setError: (error: string | null) => void;

    // Bulk operations
    bulkUpdateStatus: (ids: string[], status: Phone['status']) => void;
    bulkUpdatePriority: (ids: string[], priority: Phone['priority']) => void;
    clearPhones: () => void;
}

const initialFilters: FilterOptions = {
    search: '',
    priority: 'all',
    status: 'all',
    showEntries: 25,
};

export const usePhoneStore = create<PhoneState & PhoneActions>()(
    persist(
        (set, get) => ({
            // Initial state
            phones: [],
            filteredPhones: [],
            filters: initialFilters,
            selectedPhones: [],
            isLoading: false,
            error: null,

            // Phone CRUD operations
            addPhone: (phone) => {
                const newPhone: Phone = {
                    ...phone,
                    id: Date.now().toString(),
                };

                set((state) => ({
                    phones: [...state.phones, newPhone],
                }));
                // Reapply filters after adding to ensure consistency
                get().applyFilters();
            },

            updatePhone: (id, updates) => {
                set((state) => ({
                    phones: state.phones.map((phone) =>
                        phone.id === id ? { ...phone, ...updates } : phone
                    ),
                }));
                // Reapply filters after update to ensure consistency
                get().applyFilters();
            },

            deletePhone: (id) => {
                set((state) => ({
                    phones: state.phones.filter((phone) => phone.id !== id),
                    selectedPhones: state.selectedPhones.filter((phoneId) => phoneId !== id),
                }));
                // Reapply filters after deletion to ensure consistency
                get().applyFilters();
            },

            deleteSelectedPhones: () => {
                const { selectedPhones } = get();
                set((state) => ({
                    phones: state.phones.filter((phone) => !selectedPhones.includes(phone.id)),
                    selectedPhones: [],
                }));
                // Reapply filters after bulk deletion to ensure consistency
                get().applyFilters();
            },

            // Selection management
            selectPhone: (id) => {
                set((state) => ({
                    selectedPhones: [...state.selectedPhones, id],
                }));
            },

            deselectPhone: (id) => {
                set((state) => ({
                    selectedPhones: state.selectedPhones.filter((phoneId) => phoneId !== id),
                }));
            },

            selectAllPhones: () => {
                const { filteredPhones } = get();
                set({
                    selectedPhones: filteredPhones.map((phone) => phone.id),
                });
            },

            clearSelection: () => {
                set({ selectedPhones: [] });
            },

            togglePhoneSelection: (id) => {
                const { selectedPhones } = get();
                if (selectedPhones.includes(id)) {
                    get().deselectPhone(id);
                } else {
                    get().selectPhone(id);
                }
            },

            // Filtering and search
            setFilters: (newFilters) => {
                set((state) => ({
                    filters: { ...state.filters, ...newFilters },
                }));
                get().applyFilters();
            },

            resetFilters: () => {
                set({ filters: initialFilters });
                get().applyFilters();
            },

            applyFilters: () => {
                const { phones, filters } = get();

                let filtered = phones;

                // Search filter
                if (filters.search) {
                    const searchTerm = filters.search.toLowerCase();
                    filtered = filtered.filter((phone) =>
                        phone.phone.toLowerCase().includes(searchTerm) ||
                        phone.owner.toLowerCase().includes(searchTerm) ||
                        phone.email.toLowerCase().includes(searchTerm)
                    );
                }

                // Priority filter
                if (filters.priority !== 'all') {
                    filtered = filtered.filter((phone) => phone.priority === filters.priority);
                }

                // Status filter
                if (filters.status !== 'all') {
                    filtered = filtered.filter((phone) => phone.status === filters.status);
                }

                set({ filteredPhones: filtered });
            },

            searchPhones: (query) => {
                get().setFilters({ search: query });
            },

            // Sorting
            sortPhones: (field, direction) => {
                const { filteredPhones } = get();
                const sorted = [...filteredPhones].sort((a, b) => {
                    let aValue = a[field];
                    let bValue = b[field];

                    if (typeof aValue === 'string' && typeof bValue === 'string') {
                        aValue = aValue.toLowerCase();
                        bValue = bValue.toLowerCase();
                    }

                    if (direction === 'asc') {
                        return aValue < bValue ? -1 : aValue > bValue ? 1 : 0;
                    } else {
                        return aValue > bValue ? -1 : aValue < bValue ? 1 : 0;
                    }
                });

                set({ filteredPhones: sorted });
            },

            // Loading and error states
            setLoading: (loading) => {
                set({ isLoading: loading });
            },

            setError: (error) => {
                set({ error });
            },

            // Bulk operations
            bulkUpdateStatus: (ids, status) => {
                set((state) => ({
                    phones: state.phones.map((phone) =>
                        ids.includes(phone.id) ? { ...phone, status } : phone
                    ),
                    filteredPhones: state.filteredPhones.map((phone) =>
                        ids.includes(phone.id) ? { ...phone, status } : phone
                    ),
                }));
            },

            bulkUpdatePriority: (ids, priority) => {
                set((state) => ({
                    phones: state.phones.map((phone) =>
                        ids.includes(phone.id) ? { ...phone, priority } : phone
                    ),
                    filteredPhones: state.filteredPhones.map((phone) =>
                        ids.includes(phone.id) ? { ...phone, priority } : phone
                    ),
                }));
            },

            clearPhones: () => {
                set({ phones: [], filteredPhones: [] });
            },
        }),
        {
            name: 'phone-store',
            partialize: (state) => ({
                phones: state.phones,
                filters: state.filters,
                selectedPhones: state.selectedPhones
            }),
            onRehydrateStorage: () => (state) => {
                if (state) {
                    // Apply filters after rehydration
                    state.applyFilters();
                }
            },
        }
    )
);
