import { create } from 'zustand';
import { persist } from 'zustand/middleware';
import { AuthState, AuthActions, User } from '@/src/types/user';

export const useAuthStore = create<AuthState & AuthActions>()(
    persist(
        (set, get) => ({
            // Initial state
            user: null,
            isAuthenticated: false,
            isLoading: false,
            error: null,

            // Actions
            login: async (email: string, password: string) => {
                set({ isLoading: true, error: null });

                try {
                    const response = await fetch('/api/auth/login', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({ email, password }),
                    });

                    const data = await response.json();

                    if (!response.ok) {
                        throw new Error(data.error || 'שגיאה בהתחברות');
                    }

                    console.log('✅ Login successful:', data.user);
                    set({
                        user: data.user,
                        isAuthenticated: true,
                        isLoading: false,
                        error: null,
                    });
                    console.log('✅ State updated, isAuthenticated:', true);
                } catch (error) {
                    set({
                        user: null,
                        isAuthenticated: false,
                        isLoading: false,
                        error: error instanceof Error ? error.message : 'שגיאה בהתחברות',
                    });
                }
            },

            register: async (userData) => {
                set({ isLoading: true, error: null });

                try {
                    const response = await fetch('/api/auth/register', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify(userData),
                    });

                    const data = await response.json();

                    if (!response.ok) {
                        throw new Error(data.error || 'שגיאה בהרשמה');
                    }

                    set({
                        user: data.user,
                        isAuthenticated: true,
                        isLoading: false,
                        error: null,
                    });
                } catch (error) {
                    set({
                        user: null,
                        isAuthenticated: false,
                        isLoading: false,
                        error: error instanceof Error ? error.message : 'שגיאה בהרשמה',
                    });
                }
            },

            logout: async () => {
                try {
                    await fetch('/api/auth/logout', {
                        method: 'POST',
                    });
                } catch (error) {
                    console.error('Error during logout:', error);
                } finally {
                    set({
                        user: null,
                        isAuthenticated: false,
                        isLoading: false,
                        error: null,
                    });
                }
            },

            forgotPassword: async (email: string) => {
                set({ isLoading: true, error: null });

                try {
                    const response = await fetch('/api/auth/forgot-password', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({ email }),
                    });

                    const data = await response.json();

                    if (!response.ok) {
                        throw new Error(data.error || 'שגיאה בשליחת אימייל איפוס');
                    }

                    set({ isLoading: false, error: null });
                } catch (error) {
                    set({
                        isLoading: false,
                        error: error instanceof Error ? error.message : 'שגיאה בשליחת אימייל איפוס',
                    });
                }
            },

            resetPassword: async (token: string, newPassword: string) => {
                set({ isLoading: true, error: null });

                try {
                    const response = await fetch('/api/auth/reset-password', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({ token, newPassword }),
                    });

                    const data = await response.json();

                    if (!response.ok) {
                        throw new Error(data.error || 'שגיאה באיפוס סיסמה');
                    }

                    set({ isLoading: false, error: null });
                } catch (error) {
                    set({
                        isLoading: false,
                        error: error instanceof Error ? error.message : 'שגיאה באיפוס סיסמה',
                    });
                }
            },

            clearError: () => {
                set({ error: null });
            },

            checkAuth: async () => {
                set({ isLoading: true });

                try {
                    const response = await fetch('/api/auth/me', {
                        method: 'GET',
                        credentials: 'include',
                    });

                    if (response.ok) {
                        const data = await response.json();
                        set({
                            user: data.user,
                            isAuthenticated: true,
                            isLoading: false,
                            error: null,
                        });
                    } else {
                        set({
                            user: null,
                            isAuthenticated: false,
                            isLoading: false,
                            error: null,
                        });
                    }
                } catch (error) {
                    set({
                        user: null,
                        isAuthenticated: false,
                        isLoading: false,
                        error: null,
                    });
                }
            },
        }),
        {
            name: 'auth-storage',
            partialize: (state) => ({
                user: state.user,
                isAuthenticated: state.isAuthenticated,
            }),
        }
    )
);
