/**
 * Notes Helper Functions
 * פונקציות עזר למודול הפתקים
 */

import { Note, SmartList } from '@/types/notes';
import { addDays, isToday, isThisWeek, isThisMonth, isPast } from 'date-fns';

// Smart Lists Definitions
export const smartLists: SmartList[] = [
    {
        id: 'today',
        name: '☀️ היום שלי',
        icon: '☀️',
        color: '#FF9500',
        filter: (notes: Note[]) => {
            const today = new Date().toISOString().split('T')[0]; // YYYY-MM-DD
            return notes.filter(n => {
                if (n.done) return false;
                // כולל פתקים עם תזכורת להיום או תאריך יעד להיום
                return (n.reminderDate === today) || (n.dueDate === today);
            });
        },
    },
    {
        id: 'important',
        name: 'חשוב',
        icon: '⭐',
        color: '#F39C12',
        filter: (notes: Note[]) => notes.filter(n => n.important && !n.done),
    },
    {
        id: 'planned',
        name: 'מתוכנן',
        icon: '📅',
        color: '#4A90E2',
        filter: (notes: Note[]) => notes.filter(n => n.dueDate && !n.done),
    },
    {
        id: 'all',
        name: 'הכל',
        icon: '📋',
        color: '#95A5A6',
        filter: (notes: Note[]) => notes,
    },
    {
        id: 'completed',
        name: 'הושלמו',
        icon: '✅',
        color: '#27AE60',
        filter: (notes: Note[]) => notes.filter(n => n.done),
    },
    {
        id: 'flagged',
        name: 'דוא״ל מסומן בדגל',
        icon: '📧',
        color: '#E74C3C',
        filter: (notes: Note[]) => notes.filter(n => n.flagged),
    },
];

// Generate UUID
export function generateId(): string {
    return Date.now().toString() + Math.random().toString(36).substr(2, 9);
}

// Sort notes
export function sortNotes(notes: Note[], sortBy: string): Note[] {
    const sorted = [...notes];

    switch (sortBy) {
        case 'date':
            return sorted.sort((a, b) =>
                new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime()
            );

        case 'importance':
            return sorted.sort((a, b) => {
                if (a.important && !b.important) return -1;
                if (!a.important && b.important) return 1;
                return new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime();
            });

        case 'alphabetical':
            return sorted.sort((a, b) => a.title.localeCompare(b.title, 'he'));

        case 'dueDate':
            return sorted.sort((a, b) => {
                if (!a.dueDate && !b.dueDate) return 0;
                if (!a.dueDate) return 1;
                if (!b.dueDate) return -1;
                return new Date(a.dueDate).getTime() - new Date(b.dueDate).getTime();
            });

        default:
            return sorted;
    }
}

// Filter notes
export function filterNotes(notes: Note[], filterType: string): Note[] {
    switch (filterType) {
        case 'open':
            return notes.filter(n => !n.done);
        case 'completed':
            return notes.filter(n => n.done);
        default:
            return notes;
    }
}

// Search notes
export function searchNotes(notes: Note[], query: string): Note[] {
    if (!query.trim()) return notes;

    const lowerQuery = query.toLowerCase();

    return notes.filter(note => {
        const titleMatch = note.title.toLowerCase().includes(lowerQuery);
        const descMatch = note.description?.toLowerCase().includes(lowerQuery);
        const tagsMatch = note.tags?.some(tag => tag.toLowerCase().includes(lowerQuery));

        return titleMatch || descMatch || tagsMatch;
    });
}

// Due date helpers
export function getDueDateStatus(dueDate: string | null | undefined):
    'overdue' | 'today' | 'thisWeek' | 'later' | null {
    if (!dueDate) return null;

    const date = new Date(dueDate);

    if (isPast(date) && !isToday(date)) return 'overdue';
    if (isToday(date)) return 'today';
    if (isThisWeek(date)) return 'thisWeek';

    return 'later';
}

// Format date for display
export function formatDueDate(dueDate: string | null | undefined): string {
    if (!dueDate) return '';

    const date = new Date(dueDate);
    const status = getDueDateStatus(dueDate);

    if (status === 'today') return 'היום';
    if (status === 'overdue') return 'באיחור';

    return new Intl.DateTimeFormat('he-IL', {
        day: 'numeric',
        month: 'short',
    }).format(date);
}

// Count open notes
export function countOpenNotes(notes: Note[]): number {
    return notes.filter(n => !n.done).length;
}

