/**
 * Notes Database Connection
 * חיבור למסד נתונים של פתקים
 */

import sqlite3 from 'sqlite3';
import { open, Database } from 'sqlite';
import path from 'path';

let db: Database | null = null;

export async function getNotesDatabase(): Promise<Database> {
  if (db) return db;

  const dbPath = path.join(process.cwd(), 'notes.db');

  db = await open({
    filename: dbPath,
    driver: sqlite3.Database,
  });

  console.log('✅ Connected to Notes SQLite database at:', dbPath);

  await initializeTables();

  return db;
}

async function initializeTables() {
  if (!db) return;

  // Lists Table
  await db.exec(`
    CREATE TABLE IF NOT EXISTS lists (
      id TEXT PRIMARY KEY,
      name TEXT NOT NULL,
      color TEXT,
      icon TEXT,
      sort_order INTEGER NOT NULL DEFAULT 0,
      created_at TEXT NOT NULL,
      updated_at TEXT NOT NULL
    )
  `);

  // Notes Table
  await db.exec(`
      CREATE TABLE IF NOT EXISTS notes (
        id TEXT PRIMARY KEY,
        list_id TEXT,
        title TEXT NOT NULL,
        description TEXT,
        done INTEGER NOT NULL DEFAULT 0,
        important INTEGER NOT NULL DEFAULT 0,
        flagged INTEGER NOT NULL DEFAULT 0,
        due_date TEXT,
        reminder_date TEXT,
        reminder_time TEXT,
        reminder_recurrence TEXT DEFAULT 'none',
        created_at TEXT NOT NULL,
        updated_at TEXT NOT NULL,
        subtasks TEXT,
        tags TEXT,
        attachments TEXT,
        FOREIGN KEY (list_id) REFERENCES lists(id) ON DELETE SET NULL
      )
    `);

  // Indexes for performance
  await db.exec(`
    CREATE INDEX IF NOT EXISTS idx_notes_list_id ON notes(list_id);
    CREATE INDEX IF NOT EXISTS idx_notes_done ON notes(done);
    CREATE INDEX IF NOT EXISTS idx_notes_important ON notes(important);
    CREATE INDEX IF NOT EXISTS idx_notes_due_date ON notes(due_date);
    CREATE INDEX IF NOT EXISTS idx_notes_flagged ON notes(flagged);
    CREATE INDEX IF NOT EXISTS idx_notes_reminder_date ON notes(reminder_date);
  `);

  console.log('✅ Notes tables initialized successfully');
}

export async function closeNotesDatabase() {
  if (db) {
    await db.close();
    db = null;
    console.log('✅ Notes database connection closed');
  }
}

