import sqlite3 from 'sqlite3';
import { Database } from 'sqlite3';
import { Phone } from '@/types/phone';
import path from 'path';

let db: Database | null = null;

export const getDatabase = (): Promise<Database> => {
    return new Promise((resolve, reject) => {
        if (db) {
            resolve(db);
            return;
        }

        const dbPath = path.join(process.cwd(), 'phones.db');
        const database = new sqlite3.Database(dbPath, (err) => {
            if (err) {
                console.error('Error opening database:', err);
                reject(err);
            } else {
                console.log('Connected to SQLite database at:', dbPath);
                db = database;
                resolve(database);
            }
        });
    });
};

export const initDatabase = async (): Promise<void> => {
    const database = await getDatabase();

    return new Promise((resolve, reject) => {
        const createTableSQL = `
      CREATE TABLE IF NOT EXISTS phones (
        id TEXT PRIMARY KEY,
        phone TEXT NOT NULL,
        domain TEXT NOT NULL,
        website TEXT,
        owner TEXT NOT NULL,
        email TEXT NOT NULL,
        package TEXT NOT NULL,
        amount REAL NOT NULL,
        status TEXT NOT NULL CHECK (status IN ('pending', 'paid', 'overdue', 'free', 'paused', 'delete_backup', 'delete_no_backup', 'renewal')),
        priority TEXT NOT NULL CHECK (priority IN ('low', 'medium', 'high', 'to-delete', 'to-check')),
        startDate TEXT NOT NULL,
        nextPayment TEXT NOT NULL,
        notes TEXT,
        createdAt TEXT NOT NULL,
        updatedAt TEXT NOT NULL
      )
    `;

        database.run(createTableSQL, (err) => {
            if (err) {
                console.error('Error creating table:', err);
                reject(err);
                return;
            }

            // Ensure startDate column exists for older databases
            database.all("PRAGMA table_info(phones)", (pragmaErr, rows: any[]) => {
                if (pragmaErr) {
                    console.error('Error reading table info:', pragmaErr);
                    resolve();
                    return;
                }

                const hasStartDate = rows.some((r) => r.name === 'startDate');
                if (!hasStartDate) {
                    database.run("ALTER TABLE phones ADD COLUMN startDate TEXT", (alterErr) => {
                        if (alterErr) {
                            console.error('Error adding startDate column:', alterErr);
                        } else {
                            console.log('startDate column added to phones table');
                        }
                        // continue to status constraint check after adding column
                        // fallthrough to check constraint logic below
                    });
                }

                // Ensure status constraint supports new values by recreating table if needed
                database.get("SELECT sql FROM sqlite_master WHERE type='table' AND name='phones'", (sqlErr, row: any) => {
                    if (sqlErr || !row || typeof row.sql !== 'string') {
                        resolve();
                        return;
                    }
                    const hasNewStatuses = row.sql.includes("delete_no_backup") && row.sql.includes("paused") && row.sql.includes("renewal");
                    if (hasNewStatuses) {
                        resolve();
                        return;
                    }

                    console.log('Updating phones.status CHECK constraint to include new values...');
                    database.serialize(() => {
                        database.run("BEGIN TRANSACTION");
                        database.run("ALTER TABLE phones RENAME TO phones_old");
                        database.run(createTableSQL);
                        database.run("INSERT INTO phones (id, phone, domain, website, owner, email, package, amount, status, priority, startDate, nextPayment, notes, createdAt, updatedAt) SELECT id, phone, domain, website, owner, email, package, amount, status, priority, COALESCE(startDate, date('now')), nextPayment, notes, createdAt, updatedAt FROM phones_old");
                        database.run("DROP TABLE phones_old");
                        database.run("COMMIT", () => {
                            console.log('phones table constraint updated successfully');
                            resolve();
                        });
                    });
                });
            });
            // close database.run callback
        });
        // close Promise wrapper
    });
};

export const closeDatabase = (): Promise<void> => {
    return new Promise((resolve, reject) => {
        if (db) {
            db.close((err) => {
                if (err) {
                    console.error('Error closing database:', err);
                    reject(err);
                } else {
                    console.log('Database connection closed');
                    db = null;
                    resolve();
                }
            });
        } else {
            resolve();
        }
    });
};
