import Database from 'better-sqlite3';
import path from 'path';

let crmDatabase: Database.Database | null = null;

export const getCrmDatabase = (): Database.Database => {
    if (!crmDatabase) {
        const dbPath = path.join(process.cwd(), 'crm.db');
        crmDatabase = new Database(dbPath);
        console.log(`Connected to CRM SQLite database at: ${dbPath}`);
    }
    return crmDatabase;
};

export const initCrmDatabase = async (): Promise<void> => {
    const database = getCrmDatabase();

    return new Promise((resolve, reject) => {
        // Create customers table
        const createCustomersTableSQL = `
      CREATE TABLE IF NOT EXISTS customers (
        id TEXT PRIMARY KEY,
        name TEXT NOT NULL,
        email TEXT UNIQUE,
        phone TEXT,
        company TEXT,
        status TEXT NOT NULL CHECK (status IN ('active', 'inactive', 'lead')) DEFAULT 'lead',
        stage TEXT NOT NULL CHECK (stage IN ('quote', 'project', 'follow_up', 'closed')) DEFAULT 'quote',
        assigned_sales_rep TEXT,
        value REAL DEFAULT 0,
        notes TEXT,
        created_date TEXT NOT NULL,
        last_activity TEXT NOT NULL,
        updated_at TEXT NOT NULL
      )
    `;

        // Create quotes table
        const createQuotesTableSQL = `
      CREATE TABLE IF NOT EXISTS quotes (
        id TEXT PRIMARY KEY,
        customer_id TEXT NOT NULL,
        title TEXT NOT NULL,
        amount REAL NOT NULL,
        status TEXT NOT NULL CHECK (status IN ('pending', 'sent', 'approved', 'rejected')) DEFAULT 'pending',
        created_date TEXT NOT NULL,
        valid_until TEXT,
        notes TEXT,
        FOREIGN KEY (customer_id) REFERENCES customers (id) ON DELETE CASCADE
      )
    `;

        // Create projects table
        const createProjectsTableSQL = `
      CREATE TABLE IF NOT EXISTS projects (
        id TEXT PRIMARY KEY,
        customer_id TEXT NOT NULL,
        title TEXT NOT NULL,
        description TEXT,
        status TEXT NOT NULL CHECK (status IN ('planning', 'in_progress', 'completed', 'on_hold')) DEFAULT 'planning',
        start_date TEXT,
        end_date TEXT,
        budget REAL,
        progress INTEGER DEFAULT 0,
        created_date TEXT NOT NULL,
        updated_at TEXT NOT NULL,
        FOREIGN KEY (customer_id) REFERENCES customers (id) ON DELETE CASCADE
      )
    `;

        try {
            database.exec(createCustomersTableSQL);
            console.log('Customers table created successfully or already exists');

            database.exec(createQuotesTableSQL);
            console.log('Quotes table created successfully or already exists');

            database.exec(createProjectsTableSQL);
            console.log('Projects table created successfully or already exists');

            resolve();
        } catch (err) {
            console.error('Error creating CRM tables:', err);
            reject(err);
        }
    });
};

export const clearCrmTables = async (): Promise<void> => {
    const database = getCrmDatabase();

    return new Promise((resolve, reject) => {
        try {
            database.exec('DELETE FROM projects');
            database.exec('DELETE FROM quotes');
            database.exec('DELETE FROM customers');
            console.log('Cleared all CRM tables');
            resolve();
        } catch (err) {
            console.error('Error clearing CRM tables:', err);
            reject(err);
        }
    });
};
