import Database from 'better-sqlite3';
import path from 'path';

let db: Database.Database | null = null;

export function getComprehensiveDatabase(): Database.Database {
    if (!db) {
        const dbPath = path.join(process.cwd(), 'comprehensive.db');
        db = new Database(dbPath);
        console.log(`Connected to Comprehensive SQLite database at: ${dbPath}`);
    }
    return db;
}

export async function initComprehensiveDatabase(): Promise<void> {
    const database = getComprehensiveDatabase();

    // Enable foreign key constraints
    database.pragma('foreign_keys = ON');

    // Create all tables with proper relationships

    // 1. CUSTOMERS table (main entity)
    database.exec(`
        CREATE TABLE IF NOT EXISTS customers (
            id TEXT PRIMARY KEY,
            name TEXT NOT NULL,
            email TEXT UNIQUE,
            phone TEXT,
            company TEXT,
            status TEXT CHECK(status IN ('active', 'inactive', 'lead')) DEFAULT 'lead',
            stage TEXT CHECK(stage IN ('quote', 'project', 'follow_up', 'closed')) DEFAULT 'quote',
            assigned_sales_rep TEXT,
            value REAL DEFAULT 0,
            notes TEXT,
            created_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            last_activity DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
        )
    `);
    console.log('Customers table created successfully or already exists');

    // 2. DOMAINS table (linked to customers)
    database.exec(`
        CREATE TABLE IF NOT EXISTS domains (
            id TEXT PRIMARY KEY,
            customer_id TEXT NOT NULL,
            domain_name TEXT NOT NULL,
            website TEXT,
            owner TEXT,
            email TEXT,
            package TEXT,
            amount REAL,
            status TEXT CHECK(status IN ('active', 'inactive', 'pending', 'expired')) DEFAULT 'active',
            priority TEXT CHECK(priority IN ('low', 'medium', 'high')) DEFAULT 'medium',
            next_payment TEXT,
            last_update TEXT,
            notes TEXT,
            created_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (customer_id) REFERENCES customers (id) ON DELETE CASCADE
        )
    `);
    console.log('Domains table created successfully or already exists');

    // 3. PROJECTS table (linked to customers)
    database.exec(`
        CREATE TABLE IF NOT EXISTS projects (
            id TEXT PRIMARY KEY,
            customer_id TEXT NOT NULL,
            title TEXT NOT NULL,
            description TEXT,
            status TEXT CHECK(status IN ('planning', 'in_progress', 'completed', 'on_hold', 'cancelled')) DEFAULT 'planning',
            start_date DATE,
            end_date DATE,
            budget REAL,
            progress INTEGER DEFAULT 0,
            assigned_team TEXT,
            notes TEXT,
            created_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (customer_id) REFERENCES customers (id) ON DELETE CASCADE
        )
    `);
    console.log('Projects table created successfully or already exists');

    // 4. QUOTES table (linked to customers)
    database.exec(`
        CREATE TABLE IF NOT EXISTS quotes (
            id TEXT PRIMARY KEY,
            customer_id TEXT NOT NULL,
            title TEXT NOT NULL,
            amount REAL NOT NULL,
            status TEXT CHECK(status IN ('pending', 'sent', 'approved', 'rejected', 'expired')) DEFAULT 'pending',
            valid_until DATETIME,
            description TEXT,
            terms TEXT,
            notes TEXT,
            created_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (customer_id) REFERENCES customers (id) ON DELETE CASCADE
        )
    `);
    console.log('Quotes table created successfully or already exists');

    // 5. PAYMENTS table (linked to customers, domains, projects, quotes)
    database.exec(`
        CREATE TABLE IF NOT EXISTS payments (
            id TEXT PRIMARY KEY,
            customer_id TEXT NOT NULL,
            domain_id TEXT,
            project_id TEXT,
            quote_id TEXT,
            amount REAL NOT NULL,
            payment_type TEXT CHECK(payment_type IN ('domain_renewal', 'project_payment', 'quote_payment', 'service_payment', 'other')) NOT NULL,
            status TEXT CHECK(status IN ('pending', 'completed', 'failed', 'cancelled', 'refunded')) DEFAULT 'pending',
            payment_method TEXT CHECK(payment_method IN ('credit_card', 'bank_transfer', 'paypal', 'cash', 'check', 'other')) DEFAULT 'credit_card',
            payment_date DATETIME,
            due_date DATETIME,
            invoice_number TEXT,
            description TEXT,
            notes TEXT,
            created_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (customer_id) REFERENCES customers (id) ON DELETE CASCADE,
            FOREIGN KEY (domain_id) REFERENCES domains (id) ON DELETE SET NULL,
            FOREIGN KEY (project_id) REFERENCES projects (id) ON DELETE SET NULL,
            FOREIGN KEY (quote_id) REFERENCES quotes (id) ON DELETE SET NULL
        )
    `);
    console.log('Payments table created successfully or already exists');

    // 6. ACTIVITIES table (linked to customers)
    database.exec(`
        CREATE TABLE IF NOT EXISTS activities (
            id TEXT PRIMARY KEY,
            customer_id TEXT NOT NULL,
            activity_type TEXT CHECK(activity_type IN ('call', 'email', 'meeting', 'note', 'task', 'reminder', 'payment', 'other')) NOT NULL,
            title TEXT NOT NULL,
            description TEXT,
            status TEXT CHECK(status IN ('pending', 'completed', 'cancelled')) DEFAULT 'pending',
            priority TEXT CHECK(priority IN ('low', 'medium', 'high')) DEFAULT 'medium',
            due_date DATETIME,
            completed_date DATETIME,
            assigned_to TEXT,
            duration_minutes INTEGER,
            outcome TEXT,
            notes TEXT,
            created_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (customer_id) REFERENCES customers (id) ON DELETE CASCADE
        )
    `);
    console.log('Activities table created successfully or already exists');

    // 7. REMINDERS table (linked to customers, domains, projects)
    database.exec(`
        CREATE TABLE IF NOT EXISTS reminders (
            id TEXT PRIMARY KEY,
            customer_id TEXT NOT NULL,
            domain_id TEXT,
            project_id TEXT,
            reminder_type TEXT CHECK(reminder_type IN ('domain_renewal', 'payment_due', 'project_deadline', 'follow_up', 'meeting', 'other')) NOT NULL,
            title TEXT NOT NULL,
            description TEXT,
            reminder_date DATETIME NOT NULL,
            status TEXT CHECK(status IN ('pending', 'completed', 'cancelled', 'overdue')) DEFAULT 'pending',
            priority TEXT CHECK(priority IN ('low', 'medium', 'high')) DEFAULT 'medium',
            recurring TEXT CHECK(recurring IN ('none', 'daily', 'weekly', 'monthly', 'yearly')) DEFAULT 'none',
            completed_date DATETIME,
            notes TEXT,
            created_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (customer_id) REFERENCES customers (id) ON DELETE CASCADE,
            FOREIGN KEY (domain_id) REFERENCES domains (id) ON DELETE SET NULL,
            FOREIGN KEY (project_id) REFERENCES projects (id) ON DELETE SET NULL
        )
    `);
    console.log('Reminders table created successfully or already exists');

    // 8. EMAIL_SERVICES table (linked to domains)
    database.exec(`
        CREATE TABLE IF NOT EXISTS email_services (
            id TEXT PRIMARY KEY,
            domain_id TEXT NOT NULL,
            service_type TEXT CHECK(service_type IN ('hosting', 'email_accounts', 'ssl_certificate', 'backup', 'security', 'other')) NOT NULL,
            service_name TEXT NOT NULL,
            provider TEXT,
            status TEXT CHECK(status IN ('active', 'inactive', 'pending', 'suspended')) DEFAULT 'active',
            start_date DATE,
            end_date DATE,
            monthly_cost REAL,
            features TEXT,
            notes TEXT,
            created_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (domain_id) REFERENCES domains (id) ON DELETE CASCADE
        )
    `);
    console.log('Email services table created successfully or already exists');

    // 9. TECHNICAL_SERVICES table (linked to domains)
    database.exec(`
        CREATE TABLE IF NOT EXISTS technical_services (
            id TEXT PRIMARY KEY,
            domain_id TEXT NOT NULL,
            service_type TEXT CHECK(service_type IN ('dns_management', 'ssl_certificate', 'cdn', 'monitoring', 'backup', 'security', 'performance', 'other')) NOT NULL,
            service_name TEXT NOT NULL,
            provider TEXT,
            status TEXT CHECK(status IN ('active', 'inactive', 'pending', 'maintenance')) DEFAULT 'active',
            start_date DATE,
            end_date DATE,
            monthly_cost REAL,
            configuration TEXT,
            notes TEXT,
            created_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (domain_id) REFERENCES domains (id) ON DELETE CASCADE
        )
    `);
    console.log('Technical services table created successfully or already exists');

    // Create indexes for better performance
    database.exec(`
        CREATE INDEX IF NOT EXISTS idx_customers_email ON customers(email);
        CREATE INDEX IF NOT EXISTS idx_customers_status ON customers(status);
        CREATE INDEX IF NOT EXISTS idx_domains_customer_id ON domains(customer_id);
        CREATE INDEX IF NOT EXISTS idx_domains_status ON domains(status);
        CREATE INDEX IF NOT EXISTS idx_projects_customer_id ON projects(customer_id);
        CREATE INDEX IF NOT EXISTS idx_projects_status ON projects(status);
        CREATE INDEX IF NOT EXISTS idx_quotes_customer_id ON quotes(customer_id);
        CREATE INDEX IF NOT EXISTS idx_quotes_status ON quotes(status);
        CREATE INDEX IF NOT EXISTS idx_payments_customer_id ON payments(customer_id);
        CREATE INDEX IF NOT EXISTS idx_payments_status ON payments(status);
        CREATE INDEX IF NOT EXISTS idx_activities_customer_id ON activities(customer_id);
        CREATE INDEX IF NOT EXISTS idx_reminders_customer_id ON reminders(customer_id);
        CREATE INDEX IF NOT EXISTS idx_reminders_reminder_date ON reminders(reminder_date);
        CREATE INDEX IF NOT EXISTS idx_email_services_domain_id ON email_services(domain_id);
        CREATE INDEX IF NOT EXISTS idx_technical_services_domain_id ON technical_services(domain_id);
    `);
    console.log('Database indexes created successfully');
}

export function closeComprehensiveDatabase(): void {
    if (db) {
        db.close();
        db = null;
        console.log('Comprehensive database connection closed');
    }
}
