import { SignJWT } from 'jose';
import { NextRequest, NextResponse } from 'next/server';
import { cookies } from 'next/headers';
import { findUserById } from './db-auth';

const JWT_SECRET = process.env.JWT_SECRET || 'your-super-secret-jwt-key-change-in-production';
const JWT_EXPIRES_IN = '7d';

export interface JWTPayload {
    userId: string;
    email: string;
    role: 'ADMIN' | 'USER';
}

export const generateToken = async (payload: JWTPayload): Promise<string> => {
    const secret = new TextEncoder().encode(JWT_SECRET);
    return await new SignJWT(payload)
        .setProtectedHeader({ alg: 'HS256' })
        .setIssuedAt()
        .setExpirationTime(JWT_EXPIRES_IN)
        .sign(secret);
};

export const verifyToken = async (token: string): Promise<JWTPayload | null> => {
    try {
        const { jwtVerify } = await import('jose');
        const secret = new TextEncoder().encode(JWT_SECRET);
        const { payload } = await jwtVerify(token, secret);
        return payload as JWTPayload;
    } catch (error) {
        return null;
    }
};

export const setAuthCookie = (token: string, response: NextResponse): void => {
    response.cookies.set('auth-token', token, {
        httpOnly: true,
        secure: process.env.NODE_ENV === 'production',
        sameSite: 'strict',
        maxAge: 7 * 24 * 60 * 60, // 7 days
        path: '/'
    });
};

export const clearAuthCookie = (response: NextResponse): void => {
    response.cookies.delete('auth-token');
};

export const getAuthToken = (request: NextRequest): string | null => {
    // Try to get token from cookies first
    const cookieToken = request.cookies.get('auth-token')?.value;
    if (cookieToken) {
        return cookieToken;
    }

    // Try to get token from Authorization header
    const authHeader = request.headers.get('authorization');
    if (authHeader && authHeader.startsWith('Bearer ')) {
        return authHeader.substring(7);
    }

    return null;
};

export const getCurrentUser = async (request: NextRequest) => {
    const token = getAuthToken(request);
    if (!token) {
        return null;
    }

    const payload = await verifyToken(token);
    if (!payload) {
        return null;
    }

    const user = await findUserById(payload.userId);
    return user;
};

export const requireAuth = async (request: NextRequest) => {
    const user = await getCurrentUser(request);

    if (!user) {
        return NextResponse.json(
            { error: 'אין הרשאה - נדרשת התחברות' },
            { status: 401 }
        );
    }

    return { user };
};

export const requireRole = async (request: NextRequest, allowedRoles: ('ADMIN' | 'USER')[]) => {
    const authResult = await requireAuth(request);

    if (authResult instanceof NextResponse) {
        return authResult;
    }

    const { user } = authResult;

    if (!allowedRoles.includes(user.role)) {
        return NextResponse.json(
            { error: 'אין הרשאה - תפקיד לא מתאים' },
            { status: 403 }
        );
    }

    return { user };
};

export const isAdminRoute = (pathname: string): boolean => {
    return pathname.startsWith('/admin');
};

export const isPublicRoute = (pathname: string): boolean => {
    const publicRoutes = [
        '/login',
        '/register',
        '/forgot-password',
        '/reset-password',
        '/api/auth/login',
        '/api/auth/register',
        '/api/auth/forgot-password',
        '/api/auth/reset-password',
        '/api/auth/logout',
        '/api/health'
    ];

    return publicRoutes.some(route => pathname.startsWith(route));
};

export const shouldRedirectToLogin = (pathname: string): boolean => {
    return !isPublicRoute(pathname);
};

export const generateResetToken = (): string => {
    return Math.random().toString(36).substring(2, 15) +
        Math.random().toString(36).substring(2, 15);
};

export const validatePassword = (password: string): { isValid: boolean; errors: string[] } => {
    const errors: string[] = [];

    if (password.length < 8) {
        errors.push('הסיסמה חייבת להכיל לפחות 8 תווים');
    }

    if (!/(?=.*[a-z])/.test(password)) {
        errors.push('הסיסמה חייבת להכיל לפחות אות קטנה אחת');
    }

    if (!/(?=.*[A-Z])/.test(password)) {
        errors.push('הסיסמה חייבת להכיל לפחות אות גדולה אחת');
    }

    if (!/(?=.*\d)/.test(password)) {
        errors.push('הסיסמה חייבת להכיל לפחות ספרה אחת');
    }

    return {
        isValid: errors.length === 0,
        errors
    };
};
