'use client';

import { useEffect, useRef, useState } from 'react';
import { ExternalLink, X, AlertCircle } from 'lucide-react';

interface DomainPreviewProps {
    domain: string;
    website: string;
}

export default function DomainPreview({ domain, website }: DomainPreviewProps) {
    const [isHovered, setIsHovered] = useState(false);
    const [showPreview, setShowPreview] = useState(false);
    const [hasError, setHasError] = useState(false);
    const showTimerRef = useRef<NodeJS.Timeout | null>(null);
    const hideTimerRef = useRef<NodeJS.Timeout | null>(null);
    const anchorRef = useRef<HTMLDivElement | null>(null);
    const [position, setPosition] = useState<{ top: number; left: number }>({ top: 0, left: 0 });
    const [size, setSize] = useState<{ width: number; height: number }>({ width: 420, height: 320 });

    const handleMouseEnter = () => {
        setIsHovered(true);
        if (hideTimerRef.current) {
            clearTimeout(hideTimerRef.current);
            hideTimerRef.current = null;
        }
        if (!showTimerRef.current) {
            showTimerRef.current = setTimeout(() => {
                computePosition();
                setShowPreview(true);
                showTimerRef.current = null;
            }, 300); // small delay to avoid accidental hover
        }
    };

    const handleMouseLeave = () => {
        setIsHovered(false);
        if (showTimerRef.current) {
            clearTimeout(showTimerRef.current);
            showTimerRef.current = null;
        }
        if (!hideTimerRef.current) {
            hideTimerRef.current = setTimeout(() => {
                setShowPreview(false);
                setHasError(false);
                hideTimerRef.current = null;
            }, 250);
        }
    };

    const computePosition = () => {
        if (!anchorRef.current) return;
        const rect = anchorRef.current.getBoundingClientRect();
        // Dynamic size based on viewport
        const small = window.innerWidth < 640;
        const medium = window.innerWidth >= 640 && window.innerWidth < 1024;
        const previewWidth = small ? 360 : medium ? 420 : 520;
        const previewHeight = small ? 240 : medium ? 320 : 360;
        setSize({ width: previewWidth, height: previewHeight });
        const margin = 8;

        // Prefer above
        let top = rect.top - previewHeight - margin;
        if (top < margin) {
            // Not enough space above → place below
            top = rect.bottom + margin;
        }

        // Horizontally prefer right aligned to link end
        let left = rect.right - previewWidth;
        // Clamp to viewport
        left = Math.max(margin, Math.min(left, window.innerWidth - previewWidth - margin));

        setPosition({ top, left });
    };

    // Recompute on resize/scroll while open
    useEffect(() => {
        if (!showPreview) return;
        const onUpdate = () => computePosition();
        window.addEventListener('resize', onUpdate);
        window.addEventListener('scroll', onUpdate, true);
        return () => {
            window.removeEventListener('resize', onUpdate);
            window.removeEventListener('scroll', onUpdate, true);
        };
    }, [showPreview]);

    const handleIframeError = () => {
        setHasError(true);
    };

    const cleanWebsite = website.startsWith('http') ? website : `https://${website}`;

    return (
        <div className="relative inline-block">
            {/* Domain Link */}
            <div
                ref={anchorRef}
                onMouseEnter={handleMouseEnter}
                onMouseLeave={handleMouseLeave}
                className="relative"
            >
                <a
                    href={cleanWebsite}
                    target="_blank"
                    rel="noopener noreferrer"
                    className="text-blue-600 hover:text-blue-800 hover:underline font-medium inline-flex items-center gap-1"
                    onClick={(e) => e.stopPropagation()}
                >
                    {domain}
                    <ExternalLink className="w-3 h-3" />
                </a>
            </div>

            {/* Preview Popover (near link) */}
            {showPreview && (
                <div
                    className="fixed z-50 bg-white rounded-lg shadow-2xl border border-gray-200 overflow-hidden animate-in fade-in zoom-in-95 duration-200"
                    style={{ width: `${size.width}px`, height: `${size.height}px`, top: position.top, left: position.left }}
                    onMouseEnter={() => setIsHovered(true)}
                    onMouseLeave={handleMouseLeave}
                >
                    {/* Header */}
                    <div className="bg-gray-50 border-b border-gray-200 px-4 py-3 flex items-center justify-between">
                        <div className="flex items-center gap-2">
                            <div className="w-2 h-2 rounded-full bg-green-500"></div>
                            <span className="text-sm font-medium text-gray-700">{domain}</span>
                        </div>
                        <div className="flex items-center gap-2">
                            <a
                                href={cleanWebsite}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="p-1 text-blue-600 hover:bg-blue-50 rounded transition-colors"
                                title="פתח באתר"
                            >
                                <ExternalLink className="w-4 h-4" />
                            </a>
                            <button
                                onClick={() => setShowPreview(false)}
                                className="p-1 text-gray-400 hover:bg-gray-100 rounded transition-colors"
                                title="סגור"
                            >
                                <X className="w-4 h-4" />
                            </button>
                        </div>
                    </div>

                    {/* Preview Content */}
                    <div className="relative w-full h-full bg-gray-100">
                        {!hasError ? (
                            <>
                                <iframe
                                    src={cleanWebsite}
                                    className="w-full h-full border-0"
                                    title={`Preview of ${domain}`}
                                    sandbox="allow-same-origin allow-scripts allow-popups allow-forms"
                                    onError={handleIframeError}
                                />
                                <div className="absolute top-2 left-2 bg-black bg-opacity-50 text-white text-xs px-2 py-1 rounded">
                                    Preview
                                </div>
                            </>
                        ) : (
                            <div className="flex flex-col items-center justify-center h-full text-gray-500">
                                <AlertCircle className="w-12 h-12 mb-3 text-orange-400" />
                                <p className="text-lg font-medium mb-2">לא ניתן להציג תצוגה מקדימה</p>
                                <p className="text-sm text-gray-400 mb-4">האתר חוסם הצגה ב-iframe</p>
                                <a
                                    href={cleanWebsite}
                                    target="_blank"
                                    rel="noopener noreferrer"
                                    className="inline-flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                                >
                                    <ExternalLink className="w-4 h-4" />
                                    פתח את האתר
                                </a>
                            </div>
                        )}
                    </div>
                </div>
            )}

            {/* No backdrop for popover version */}
        </div>
    );
}

