'use client';

import { Phone } from '@/types/phone';
import { AlertTriangle, Clock, Check, Globe } from 'lucide-react';

interface StatsCardsProps {
    phones: Phone[];
}

const iconMap = {
    'triangle-warning': AlertTriangle,
    'clock': Clock,
    'check': Check,
    'globe': Globe,
};

const colorClasses: Record<string, string> = {
    red: 'bg-red-500',
    yellow: 'bg-yellow-500',
    green: 'bg-green-500',
    blue: 'bg-blue-500',
};

export default function StatsCards({ phones }: StatsCardsProps) {
    // Calculate statistics dynamically from phones data
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    const ignoreOverdueStatuses = new Set(['free', 'delete_backup', 'delete_no_backup', 'paid']);
    const isOverdue = (p: Phone) => {
        const d = new Date(p.nextPayment);
        return !isNaN(d.getTime()) && d < today && !ignoreOverdueStatuses.has(p.status);
    };
    const overdueCount = phones.filter(isOverdue).length;
    const pendingCount = phones.filter(phone => phone.status === 'pending').length;
    const paidCount = phones.filter(phone => phone.status === 'paid').length;
    const totalCount = phones.length;

    // Calculate total revenue
    const totalRevenue = phones.reduce((sum, phone) => sum + phone.amount, 0);
    const overdueRevenue = phones.filter(isOverdue).reduce((sum, phone) => sum + phone.amount, 0);

    const stats = [
        {
            id: 'overdue',
            title: 'פיגור',
            value: overdueCount,
            icon: 'triangle-warning',
            color: 'red',
            subtitle: `₪${overdueRevenue.toLocaleString()}`
        },
        {
            id: 'pending',
            title: 'ממתין',
            value: pendingCount,
            icon: 'clock',
            color: 'yellow',
            subtitle: 'תשלום ממתין'
        },
        {
            id: 'paid',
            title: 'שולם',
            value: paidCount,
            icon: 'check',
            color: 'green',
            subtitle: 'תשלום הושלם'
        },
        {
            id: 'total',
            title: 'סה"כ דומיינים',
            value: totalCount,
            icon: 'globe',
            color: 'blue',
            subtitle: `₪${totalRevenue.toLocaleString()}`
        }
    ];

    return (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
            {stats.map((stat) => {
                const IconComponent = iconMap[stat.icon as keyof typeof iconMap];
                return (
                    <div
                        key={stat.id}
                        className="bg-white rounded-lg shadow-sm border p-6 hover:shadow-md transition-shadow"
                    >
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm font-medium text-gray-600 mb-1">
                                    {stat.title}
                                </p>
                                <p className="text-3xl font-bold text-gray-900">
                                    {stat.value}
                                </p>
                                {stat.subtitle && (
                                    <p className="text-xs text-gray-500 mt-1">
                                        {stat.subtitle}
                                    </p>
                                )}
                            </div>
                            <div className={`w-12 h-12 rounded-full ${colorClasses[stat.color]} flex items-center justify-center`}>
                                <IconComponent className="w-6 h-6 text-white" />
                            </div>
                        </div>
                    </div>
                );
            })}
        </div>
    );
}