'use client';

import { useState } from 'react';
import { Phone } from '@/types/phone';
import { usePhoneStore } from '@/store/sqliteStore';
import { useUIStore } from '@/store/uiStore';
import DomainPreview from '@/src/components/domains/DomainPreview';
import {
    Edit,
    Trash2,
    MoreVertical,
    ChevronLeft,
    ChevronRight,
    Check,
    X,
    Eye
} from 'lucide-react';
import Modal from '../modals/Modal';

interface PhonesTableProps {
    phones: Phone[];
}

export default function PhonesTable({ phones }: PhonesTableProps) {
    const {
        selectedPhones,
        selectPhone,
        deselectPhone,
        selectAllPhones,
        clearSelection,
        togglePhoneSelection,
        deletePhone,
        deleteSelectedPhones,
        updatePhone
    } = usePhoneStore();

    // Helper function to parse dd/mm/yyyy format to JavaScript Date
    const parseDate = (dateString: string): Date | null => {
        if (!dateString) return null;

        // Handle dd/mm/yyyy format
        const parts = dateString.split('/');
        if (parts.length === 3) {
            const day = parseInt(parts[0], 10);
            const month = parseInt(parts[1], 10) - 1; // JavaScript months are 0-based
            const year = parseInt(parts[2], 10);
            return new Date(year, month, day);
        }

        // Fallback to standard Date parsing
        const date = new Date(dateString);
        return isNaN(date.getTime()) ? null : date;
    };

    // Helper function to format date for display
    const formatDate = (dateString: string): string => {
        const date = parseDate(dateString);
        if (!date) return '-';
        return date.toLocaleDateString('he-IL');
    };

    const {
        openEditPhoneModal,
        openDeleteConfirmModal
    } = useUIStore();

    const [currentPage, setCurrentPage] = useState(1);
    const [itemsPerPage, setItemsPerPage] = useState(10);
    const [viewPhone, setViewPhone] = useState<Phone | null>(null);

    // Inline edit state
    const [editingId, setEditingId] = useState<string | null>(null);
    const [editingField, setEditingField] = useState<'owner' | 'phone' | 'status' | null>(null);
    const [tempOwner, setTempOwner] = useState('');
    const [tempPhone, setTempPhone] = useState('');
    const [tempStatus, setTempStatus] = useState<string>('pending');

    const beginEdit = (p: Phone, field: 'owner' | 'phone' | 'status') => {
        setEditingId(p.id);
        setEditingField(field);
        if (field === 'owner') setTempOwner(p.owner);
        if (field === 'phone') setTempPhone(p.phone);
        if (field === 'status') setTempStatus(p.status);
    };

    const cancelEdit = () => {
        setEditingId(null);
        setEditingField(null);
    };

    const saveEdit = async (p: Phone) => {
        if (!editingId || !editingField) return;
        try {
            if (editingField === 'owner') {
                await updatePhone(p.id, { owner: tempOwner, lastUpdate: new Date().toISOString().split('T')[0], updatedAt: new Date().toISOString() });
            } else if (editingField === 'phone') {
                await updatePhone(p.id, { phone: tempPhone, lastUpdate: new Date().toISOString().split('T')[0], updatedAt: new Date().toISOString() });
            } else if (editingField === 'status') {
                await updatePhone(p.id, { status: tempStatus as any, lastUpdate: new Date().toISOString().split('T')[0], updatedAt: new Date().toISOString() });
            }
        } finally {
            cancelEdit();
        }
    };

    const totalPages = Math.ceil(phones.length / itemsPerPage);
    const startIndex = (currentPage - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;
    const currentPhones = phones.slice(startIndex, endIndex);

    const handleSelectAll = () => {
        if (selectedPhones.length === currentPhones.length) {
            clearSelection();
        } else {
            selectAllPhones();
        }
    };

    const handleDeleteSelected = async () => {
        if (selectedPhones.length > 0) {
            await deleteSelectedPhones();
            clearSelection();
        }
    };

    const getStatusColor = (status: string) => {
        switch (status) {
            case 'paid':
                return 'bg-green-100 text-green-800';
            case 'pending':
                return 'bg-yellow-100 text-yellow-800';
            case 'overdue':
                return 'bg-red-100 text-red-800';
            case 'free':
                return 'bg-blue-100 text-blue-800';
            case 'renewal':
                return 'bg-teal-100 text-teal-800';
            case 'paused':
                return 'bg-gray-200 text-gray-800';
            case 'delete_backup':
                return 'bg-purple-100 text-purple-800';
            case 'delete_no_backup':
                return 'bg-black text-white';
            default:
                return 'bg-gray-100 text-gray-800';
        }
    };

    const getPriorityColor = (priority: string) => {
        switch (priority) {
            case 'high':
                return 'bg-red-100 text-red-800';
            case 'medium':
                return 'bg-yellow-100 text-yellow-800';
            case 'low':
                return 'bg-green-100 text-green-800';
            case 'to-delete':
                return 'bg-red-200 text-red-900';
            case 'to-check':
                return 'bg-orange-100 text-orange-800';
            default:
                return 'bg-gray-100 text-gray-800';
        }
    };

    const getStatusText = (status: string) => {
        switch (status) {
            case 'paid':
                return 'שולם';
            case 'pending':
                return 'ממתין';
            case 'overdue':
                return 'פיגור';
            case 'free':
                return 'חינם';
            case 'renewal':
                return 'לחידוש';
            case 'paused':
                return 'מושהה';
            case 'delete_backup':
                return 'למחוק (יש גיבוי)';
            case 'delete_no_backup':
                return 'למחוק (אין גיבוי)';
            default:
                return status;
        }
    };

    const getPriorityText = (priority: string) => {
        switch (priority) {
            case 'high':
                return 'גבוהה';
            case 'medium':
                return 'בינונית';
            case 'low':
                return 'נמוכה';
            case 'to-delete':
                return 'למחוק';
            case 'to-check':
                return 'לבדוק';
            default:
                return priority;
        }
    };

    return (
        <div className="bg-white shadow-sm rounded-lg border border-gray-200">
            {/* Table Header */}
            <div className="px-6 py-4 border-b border-gray-200">
                <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-4 space-x-reverse">
                        <h3 className="text-lg font-medium text-gray-900">דומיינים</h3>
                        {selectedPhones.length > 0 && (
                            <div className="flex items-center space-x-2 space-x-reverse">
                                <span className="text-sm text-gray-500">
                                    {selectedPhones.length} נבחרו
                                </span>
                                <button
                                    onClick={handleDeleteSelected}
                                    className="inline-flex items-center px-3 py-1 border border-transparent text-sm font-medium rounded-md text-red-700 bg-red-100 hover:bg-red-200 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500"
                                >
                                    <Trash2 className="w-4 h-4 ml-1" />
                                    מחק נבחרים
                                </button>
                            </div>
                        )}
                    </div>

                    <div className="flex items-center space-x-2 space-x-reverse">
                        <span className="text-sm text-gray-500">הצג:</span>
                        <select
                            value={itemsPerPage}
                            onChange={(e) => setItemsPerPage(Number(e.target.value))}
                            className="block px-3 py-1 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 text-sm"
                            aria-label="בחר מספר רשומות לעמוד"
                        >
                            <option value={10}>10</option>
                            <option value={25}>25</option>
                            <option value={50}>50</option>
                            <option value={100}>100</option>
                        </select>
                    </div>
                </div>
            </div>

            {/* Table */}
            <div className="overflow-x-auto">
                <table className="min-w-full divide-y divide-gray-200">
                    <thead className="bg-gray-50">
                        <tr>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                <input
                                    type="checkbox"
                                    checked={selectedPhones.length === currentPhones.length && currentPhones.length > 0}
                                    onChange={handleSelectAll}
                                    className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                                    aria-label="בחר הכל"
                                />
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                בעלים
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                פלאפון
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                דומיין
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                אימייל
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                סכום
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                תאריך התחלה
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                תשלום הבא
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                אתר
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                סטטוס
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                עדיפות
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                פעולות
                            </th>
                        </tr>
                    </thead>
                    <tbody className="bg-white divide-y divide-gray-200">
                        {currentPhones.map((phone, index) => (
                            <tr key={`${phone.id}-${index}`} className="hover:bg-gray-50">
                                <td className="px-6 py-4 whitespace-nowrap">
                                    <input
                                        type="checkbox"
                                        checked={selectedPhones.includes(phone.id)}
                                        onChange={() => togglePhoneSelection(phone.id)}
                                        className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                                        aria-label={`בחר דומיין ${phone.domain}`}
                                    />
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    {editingId === phone.id && editingField === 'owner' ? (
                                        <input
                                            autoFocus
                                            value={tempOwner}
                                            onChange={(e) => setTempOwner(e.target.value)}
                                            onBlur={() => saveEdit(phone)}
                                            onKeyDown={(e) => {
                                                if (e.key === 'Enter') saveEdit(phone);
                                                if (e.key === 'Escape') cancelEdit();
                                            }}
                                            className="block w-full px-2 py-1 border border-gray-300 rounded"
                                        />
                                    ) : (
                                        <button
                                            onClick={() => beginEdit(phone, 'owner')}
                                            className="text-sm font-medium text-gray-900 hover:underline"
                                            title="עריכה מהירה"
                                        >
                                            {phone.owner}
                                        </button>
                                    )}
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    {editingId === phone.id && editingField === 'phone' ? (
                                        <input
                                            autoFocus
                                            value={tempPhone}
                                            onChange={(e) => setTempPhone(e.target.value)}
                                            onBlur={() => saveEdit(phone)}
                                            onKeyDown={(e) => {
                                                if (e.key === 'Enter') saveEdit(phone);
                                                if (e.key === 'Escape') cancelEdit();
                                            }}
                                            className="block w-full px-2 py-1 border border-gray-300 rounded"
                                        />
                                    ) : (
                                        <button
                                            onClick={() => beginEdit(phone, 'phone')}
                                            className="text-sm font-medium text-gray-900 hover:underline"
                                            title="עריכה מהירה"
                                        >
                                            {phone.phone}
                                        </button>
                                    )}
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    <DomainPreview
                                        domain={phone.domain}
                                        website={phone.website}
                                    />
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    <div className="text-sm text-gray-900">{phone.email}</div>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    <div className="text-sm text-gray-900">₪{phone.amount.toFixed(2)}</div>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                    {formatDate(phone.startDate)}
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                    {formatDate(phone.nextPayment)}
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    <div className="text-sm text-gray-900">
                                        <a
                                            href={phone.website}
                                            target="_blank"
                                            rel="noopener noreferrer"
                                            className="text-blue-600 hover:text-blue-800"
                                        >
                                            {phone.website}
                                        </a>
                                    </div>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    {editingId === phone.id && editingField === 'status' ? (
                                        <select
                                            autoFocus
                                            value={tempStatus}
                                            onChange={(e) => setTempStatus(e.target.value)}
                                            onBlur={() => saveEdit(phone)}
                                            className="px-2 py-1 border border-gray-300 rounded"
                                        >
                                            <option value="pending">ממתין</option>
                                            <option value="paid">שולם</option>
                                            <option value="overdue">פיגור</option>
                                            <option value="free">חינם</option>
                                            <option value="renewal">לחידוש</option>
                                            <option value="paused">מושהה</option>
                                            <option value="delete_backup">למחוק יש גיבוי</option>
                                            <option value="delete_no_backup">למחוק אין גיבוי</option>
                                        </select>
                                    ) : (
                                        (() => {
                                            const today = new Date();
                                            today.setHours(0, 0, 0, 0);
                                            const np = parseDate(phone.nextPayment);
                                            const ignoreOverdueStatuses = new Set(['free', 'delete_backup', 'delete_no_backup']);
                                            const isOverdue = np && np < today && phone.status !== 'paid' && !ignoreOverdueStatuses.has(phone.status);
                                            const statusToShow = isOverdue ? 'overdue' : phone.status;
                                            return (
                                                <button
                                                    onClick={() => beginEdit(phone, 'status')}
                                                    className="hover:opacity-80"
                                                    title="עריכה מהירה"
                                                >
                                                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(statusToShow)}`}>
                                                        {getStatusText(statusToShow)}
                                                    </span>
                                                </button>
                                            );
                                        })()
                                    )}
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getPriorityColor(phone.priority)}`}>
                                        {getPriorityText(phone.priority)}
                                    </span>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                    <div className="flex items-center space-x-2 space-x-reverse">
                                        <button
                                            onClick={() => openEditPhoneModal(phone.id)}
                                            className="text-blue-600 hover:text-blue-900"
                                            title="ערוך"
                                            aria-label="ערוך דומיין"
                                        >
                                            <Edit className="w-4 h-4" />
                                        </button>
                                        <button
                                            onClick={() => setViewPhone(phone)}
                                            className="text-gray-700 hover:text-gray-900"
                                            title="צפייה"
                                            aria-label="צפייה"
                                        >
                                            <Eye className="w-4 h-4" />
                                        </button>
                                        <button
                                            onClick={() => openDeleteConfirmModal(phone.id)}
                                            className="text-red-600 hover:text-red-900"
                                            title="מחק"
                                            aria-label="מחק דומיין"
                                        >
                                            <Trash2 className="w-4 h-4" />
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        ))}
                    </tbody>
                </table>
            </div>

            {/* View-only Modal */}
            <Modal
                isOpen={!!viewPhone}
                onClose={() => setViewPhone(null)}
                title={viewPhone ? `צפייה בדומיין: ${viewPhone.domain}` : ''}
                size="lg"
            >
                {viewPhone && (
                    <div className="p-6">
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <div className="text-sm text-gray-500">בעלים</div>
                                <div className="font-medium">{viewPhone.owner}</div>
                            </div>
                            <div>
                                <div className="text-sm text-gray-500">פלאפון</div>
                                <div className="font-medium">{viewPhone.phone}</div>
                            </div>
                            <div className="md:col-span-2">
                                <div className="text-sm text-gray-500">דומיין</div>
                                <div className="font-medium">{viewPhone.domain}</div>
                            </div>
                            <div className="md:col-span-2">
                                <div className="text-sm text-gray-500">אתר</div>
                                <div className="font-medium break-all">
                                    <a href={viewPhone.website} target="_blank" rel="noopener noreferrer" className="text-blue-600 hover:text-blue-800">
                                        {viewPhone.website}
                                    </a>
                                </div>
                            </div>
                            <div>
                                <div className="text-sm text-gray-500">אימייל</div>
                                <div className="font-medium">{viewPhone.email}</div>
                            </div>
                            <div>
                                <div className="text-sm text-gray-500">חבילה</div>
                                <div className="font-medium">{viewPhone.package}</div>
                            </div>
                            <div>
                                <div className="text-sm text-gray-500">סכום</div>
                                <div className="font-medium">₪{viewPhone.amount.toFixed(2)}</div>
                            </div>
                            <div>
                                <div className="text-sm text-gray-500">סטטוס</div>
                                <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(viewPhone.status)}`}>
                                    {getStatusText(viewPhone.status)}
                                </span>
                            </div>
                            <div>
                                <div className="text-sm text-gray-500">עדיפות</div>
                                <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getPriorityColor(viewPhone.priority)}`}>
                                    {getPriorityText(viewPhone.priority)}
                                </span>
                            </div>
                            <div>
                                <div className="text-sm text-gray-500">תאריך התחלה</div>
                                <div className="font-medium">{viewPhone.startDate ? new Date(viewPhone.startDate).toLocaleDateString('he-IL') : '-'}</div>
                            </div>
                            <div>
                                <div className="text-sm text-gray-500">תשלום הבא</div>
                                <div className="font-medium">{new Date(viewPhone.nextPayment).toLocaleDateString('he-IL')}</div>
                            </div>
                            <div className="md:col-span-2">
                                <div className="text-sm text-gray-500">הערות</div>
                                <div className="font-medium whitespace-pre-wrap">{viewPhone.notes || '—'}</div>
                            </div>
                        </div>
                        <div className="mt-6 flex justify-end">
                            <button
                                onClick={() => setViewPhone(null)}
                                className="px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-md hover:bg-gray-50"
                            >
                                סגור
                            </button>
                        </div>
                    </div>
                )}
            </Modal>

            {/* Pagination */}
            {totalPages > 1 && (
                <div className="px-6 py-4 border-t border-gray-200">
                    <div className="flex items-center justify-between">
                        <div className="flex items-center text-sm text-gray-700">
                            <span>
                                מציג {startIndex + 1} עד {Math.min(endIndex, phones.length)} מתוך {phones.length} דומיינים
                            </span>
                        </div>
                        <div className="flex items-center space-x-2 space-x-reverse">
                            <button
                                onClick={() => setCurrentPage(prev => Math.max(prev - 1, 1))}
                                disabled={currentPage === 1}
                                className="relative inline-flex items-center px-3 py-2 border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
                                title="עמוד קודם"
                                aria-label="עמוד קודם"
                            >
                                <ChevronRight className="w-4 h-4" />
                            </button>

                            <div className="flex items-center space-x-1 space-x-reverse">
                                {Array.from({ length: totalPages }, (_, i) => i + 1).map(page => (
                                    <button
                                        key={page}
                                        onClick={() => setCurrentPage(page)}
                                        className={`relative inline-flex items-center px-4 py-2 border text-sm font-medium ${page === currentPage
                                            ? 'z-10 bg-blue-50 border-blue-500 text-blue-600'
                                            : 'bg-white border-gray-300 text-gray-500 hover:bg-gray-50'
                                            }`}
                                    >
                                        {page}
                                    </button>
                                ))}
                            </div>

                            <button
                                onClick={() => setCurrentPage(prev => Math.min(prev + 1, totalPages))}
                                disabled={currentPage === totalPages}
                                className="relative inline-flex items-center px-3 py-2 border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
                                title="עמוד הבא"
                                aria-label="עמוד הבא"
                            >
                                <ChevronLeft className="w-4 h-4" />
                            </button>
                        </div>
                    </div>
                </div>
            )}
        </div>
    );
}