'use client';

import { Phone } from '@/types/phone';
import { X, AlertTriangle } from 'lucide-react';
import { useAlertStore } from '@/store/alertStore';

interface AlertBannersProps {
    phones: Phone[];
}

export default function AlertBanners({ phones }: AlertBannersProps) {
    const { dismissAlert, isAlertDismissed } = useAlertStore();

    // Calculate dynamic alerts based on phone data
    const pendingCount = phones.filter(phone => phone.status === 'pending').length;
    const overdueCount = phones.filter(phone => phone.status === 'overdue').length;
    const highPriorityCount = phones.filter(phone => phone.priority === 'high').length;

    // Calculate overdue revenue
    const overdueRevenue = phones
        .filter(phone => phone.status === 'overdue')
        .reduce((sum, phone) => sum + phone.amount, 0);

    // Calculate pending revenue
    const pendingRevenue = phones
        .filter(phone => phone.status === 'pending')
        .reduce((sum, phone) => sum + phone.amount, 0);

    const alerts = [
        ...(pendingCount > 0 && !isAlertDismissed('payments-pending') ? [{
            id: 'payments-pending',
            type: 'warning' as const,
            title: 'תשלומים ממתינים',
            message: `${pendingCount} דומיינים ממתינים לתשלום`,
            count: pendingCount,
            revenue: pendingRevenue
        }] : []),
        ...(overdueCount > 0 && !isAlertDismissed('overdue-payments') ? [{
            id: 'overdue-payments',
            type: 'error' as const,
            title: 'פיגורים בתשלום',
            message: `${overdueCount} דומיינים בפיגור - נדרש טיפול דחוף`,
            count: overdueCount,
            revenue: overdueRevenue
        }] : []),
        ...(highPriorityCount > 0 && !isAlertDismissed('high-priority') ? [{
            id: 'high-priority',
            type: 'info' as const,
            title: 'דומיינים בעדיפות גבוהה',
            message: `${highPriorityCount} דומיינים דורשים תשומת לב מיוחדת`,
            count: highPriorityCount,
            revenue: 0
        }] : [])
    ];

    if (alerts.length === 0) return null;

    return (
        <div className="mb-8">
            <div className="flex items-center mb-4">
                <AlertTriangle className="w-5 h-5 text-gray-600 ml-2" />
                <h3 className="text-lg font-semibold text-gray-900">התראות</h3>
            </div>

            <div className="space-y-3">
                {alerts.map((alert) => (
                    <div
                        key={alert.id}
                        className={`p-4 rounded-lg border-r-4 flex items-center justify-between ${alert.type === 'error'
                            ? 'bg-red-50 border-red-400 text-red-800'
                            : alert.type === 'warning'
                                ? 'bg-yellow-50 border-yellow-400 text-yellow-800'
                                : 'bg-blue-50 border-blue-400 text-blue-800'
                            }`}
                    >
                        <div className="flex items-center">
                            <div className="flex-1">
                                <h4 className="font-semibold text-sm">{alert.title}</h4>
                                <p className="text-sm mt-1">{alert.message}</p>
                                {alert.revenue > 0 && (
                                    <p className="text-xs mt-1 opacity-75">
                                        סכום כולל: ₪{alert.revenue.toLocaleString()}
                                    </p>
                                )}
                            </div>
                        </div>
                        <button
                            className="text-gray-400 hover:text-gray-600 transition-colors"
                            title="סגור התראה"
                            aria-label="סגור התראה"
                            onClick={() => dismissAlert(alert.id)}
                        >
                            <X className="w-4 h-4" />
                        </button>
                    </div>
                ))}
            </div>
        </div>
    );
}