'use client';

import { Phone } from '@/types/phone';
import { TrendingUp, Calendar, DollarSign, AlertCircle } from 'lucide-react';

interface AdvancedStatsProps {
    phones: Phone[];
}

export default function AdvancedStats({ phones }: AdvancedStatsProps) {
    // Calculate advanced statistics
    const totalRevenue = phones.reduce((sum, phone) => sum + phone.amount, 0);
    const averageRevenue = phones.length > 0 ? totalRevenue / phones.length : 0;

    // Revenue by status
    const paidRevenue = phones
        .filter(phone => phone.status === 'paid')
        .reduce((sum, phone) => sum + phone.amount, 0);

    const pendingRevenue = phones
        .filter(phone => phone.status === 'pending')
        .reduce((sum, phone) => sum + phone.amount, 0);

    const overdueRevenue = phones
        .filter(phone => phone.status === 'overdue')
        .reduce((sum, phone) => sum + phone.amount, 0);

    // Priority distribution
    const highPriority = phones.filter(phone => phone.priority === 'high').length;
    const mediumPriority = phones.filter(phone => phone.priority === 'medium').length;
    const lowPriority = phones.filter(phone => phone.priority === 'low').length;

    // Upcoming payments (next 30 days)
    const today = new Date();
    const nextMonth = new Date(today.getTime() + 30 * 24 * 60 * 60 * 1000);

    const upcomingPayments = phones.filter(phone => {
        const paymentDate = new Date(phone.nextPayment);
        return paymentDate >= today && paymentDate <= nextMonth;
    }).length;

    const upcomingRevenue = phones
        .filter(phone => {
            const paymentDate = new Date(phone.nextPayment);
            return paymentDate >= today && paymentDate <= nextMonth;
        })
        .reduce((sum, phone) => sum + phone.amount, 0);

    const stats = [
        {
            id: 'total-revenue',
            title: 'סה"כ הכנסות',
            value: `₪${totalRevenue.toLocaleString()}`,
            icon: DollarSign,
            color: 'text-green-600',
            bgColor: 'bg-green-50',
            change: '+12.5%',
            changeType: 'positive'
        },
        {
            id: 'average-revenue',
            title: 'הכנסה ממוצעת',
            value: `₪${Math.round(averageRevenue).toLocaleString()}`,
            icon: TrendingUp,
            color: 'text-blue-600',
            bgColor: 'bg-blue-50',
            change: '+8.2%',
            changeType: 'positive'
        },
        {
            id: 'upcoming-payments',
            title: 'תשלומים קרובים',
            value: upcomingPayments,
            icon: Calendar,
            color: 'text-purple-600',
            bgColor: 'bg-purple-50',
            subtitle: `₪${upcomingRevenue.toLocaleString()}`,
            change: '30 יום',
            changeType: 'neutral'
        },
        {
            id: 'high-priority',
            title: 'עדיפות גבוהה',
            value: highPriority,
            icon: AlertCircle,
            color: 'text-red-600',
            bgColor: 'bg-red-50',
            subtitle: `${Math.round((highPriority / phones.length) * 100)}%`,
            change: 'דחוף',
            changeType: 'negative'
        }
    ];

    return (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
            {stats.map((stat) => {
                const IconComponent = stat.icon;
                return (
                    <div
                        key={stat.id}
                        className="bg-white rounded-lg shadow-sm border p-6 hover:shadow-md transition-shadow"
                    >
                        <div className="flex items-center justify-between">
                            <div className="flex-1">
                                <p className="text-sm font-medium text-gray-600 mb-1">
                                    {stat.title}
                                </p>
                                <p className="text-2xl font-bold text-gray-900 mb-1">
                                    {stat.value}
                                </p>
                                {stat.subtitle && (
                                    <p className="text-xs text-gray-500 mb-2">
                                        {stat.subtitle}
                                    </p>
                                )}
                                <div className="flex items-center">
                                    <span className={`text-xs font-medium ${stat.changeType === 'positive' ? 'text-green-600' :
                                            stat.changeType === 'negative' ? 'text-red-600' :
                                                'text-gray-600'
                                        }`}>
                                        {stat.change}
                                    </span>
                                </div>
                            </div>
                            <div className={`w-12 h-12 rounded-full ${stat.bgColor} flex items-center justify-center`}>
                                <IconComponent className={`w-6 h-6 ${stat.color}`} />
                            </div>
                        </div>
                    </div>
                );
            })}
        </div>
    );
}
