'use client';

import { useState, useRef, useEffect } from 'react';
import { Mic, Square, Play, Pause, Trash2, Upload } from 'lucide-react';

interface VoiceRecorderProps {
    entityType: 'customer' | 'task' | 'event' | 'project';
    entityId: number;
    onUploadSuccess?: () => void;
}

export default function VoiceRecorder({ entityType, entityId, onUploadSuccess }: VoiceRecorderProps) {
    const [isRecording, setIsRecording] = useState(false);
    const [isPlaying, setIsPlaying] = useState(false);
    const [audioBlob, setAudioBlob] = useState<Blob | null>(null);
    const [audioUrl, setAudioUrl] = useState<string | null>(null);
    const [duration, setDuration] = useState(0);
    const [recordingTime, setRecordingTime] = useState(0);
    const [isUploading, setIsUploading] = useState(false);

    const mediaRecorderRef = useRef<MediaRecorder | null>(null);
    const audioRef = useRef<HTMLAudioElement | null>(null);
    const chunksRef = useRef<Blob[]>([]);
    const timerRef = useRef<NodeJS.Timeout | null>(null);

    useEffect(() => {
        return () => {
            if (timerRef.current) clearInterval(timerRef.current);
            if (audioUrl) URL.revokeObjectURL(audioUrl);
        };
    }, [audioUrl]);

    const startRecording = async () => {
        try {
            const stream = await navigator.mediaDevices.getUserMedia({ audio: true });
            const mediaRecorder = new MediaRecorder(stream, {
                mimeType: 'audio/webm'
            });

            chunksRef.current = [];
            mediaRecorderRef.current = mediaRecorder;

            mediaRecorder.ondataavailable = (e) => {
                if (e.data.size > 0) {
                    chunksRef.current.push(e.data);
                }
            };

            mediaRecorder.onstop = () => {
                const blob = new Blob(chunksRef.current, { type: 'audio/webm' });
                const url = URL.createObjectURL(blob);
                setAudioBlob(blob);
                setAudioUrl(url);
                setDuration(recordingTime);

                stream.getTracks().forEach(track => track.stop());
            };

            mediaRecorder.start();
            setIsRecording(true);
            setRecordingTime(0);

            // Start timer
            timerRef.current = setInterval(() => {
                setRecordingTime(prev => prev + 1);
            }, 1000);

        } catch (error) {
            console.error('Error accessing microphone:', error);
            alert('שגיאה בגישה למיקרופון. אנא בדוק את ההרשאות.');
        }
    };

    const stopRecording = () => {
        if (mediaRecorderRef.current && isRecording) {
            mediaRecorderRef.current.stop();
            setIsRecording(false);
            if (timerRef.current) {
                clearInterval(timerRef.current);
                timerRef.current = null;
            }
        }
    };

    const togglePlayback = () => {
        if (!audioRef.current || !audioUrl) return;

        if (isPlaying) {
            audioRef.current.pause();
            setIsPlaying(false);
        } else {
            audioRef.current.play();
            setIsPlaying(true);
        }
    };

    const deleteRecording = () => {
        if (audioUrl) URL.revokeObjectURL(audioUrl);
        setAudioBlob(null);
        setAudioUrl(null);
        setDuration(0);
        setRecordingTime(0);
        setIsPlaying(false);
    };

    const uploadRecording = async () => {
        if (!audioBlob) return;

        setIsUploading(true);

        try {
            const formData = new FormData();
            const fileName = `voice-recording-${Date.now()}.webm`;
            formData.append('file', audioBlob, fileName);
            formData.append('entity_type', entityType);
            formData.append('entity_id', entityId.toString());
            formData.append('is_voice_recording', 'true');
            formData.append('duration', duration.toString());

            const response = await fetch('/api/attachments/upload', {
                method: 'POST',
                body: formData,
            });

            const data = await response.json();

            if (data.success) {
                alert('ההקלטה נשמרה בהצלחה!');
                deleteRecording();
                onUploadSuccess?.();
            } else {
                throw new Error(data.error || 'Failed to upload');
            }
        } catch (error) {
            console.error('Error uploading recording:', error);
            alert('שגיאה בשמירת ההקלטה');
        } finally {
            setIsUploading(false);
        }
    };

    const formatTime = (seconds: number) => {
        const mins = Math.floor(seconds / 60);
        const secs = seconds % 60;
        return `${mins}:${secs.toString().padStart(2, '0')}`;
    };

    return (
        <div className="flex items-center gap-2 p-3 bg-gray-50 rounded-lg border border-gray-200">
            {!audioBlob ? (
                <>
                    <button
                        onClick={isRecording ? stopRecording : startRecording}
                        className={`p-2 rounded-full transition-colors ${isRecording
                                ? 'bg-red-500 hover:bg-red-600 text-white animate-pulse'
                                : 'bg-blue-500 hover:bg-blue-600 text-white'
                            }`}
                        title={isRecording ? 'עצור הקלטה' : 'התחל הקלטה'}
                    >
                        {isRecording ? <Square size={20} /> : <Mic size={20} />}
                    </button>
                    {isRecording && (
                        <span className="text-sm font-mono text-red-600">
                            {formatTime(recordingTime)}
                        </span>
                    )}
                    <span className="text-sm text-gray-600">
                        {isRecording ? 'מקליט...' : 'לחץ להקלטה קולית'}
                    </span>
                </>
            ) : (
                <>
                    <audio
                        ref={audioRef}
                        src={audioUrl || undefined}
                        onEnded={() => setIsPlaying(false)}
                        className="hidden"
                    />
                    <button
                        onClick={togglePlayback}
                        className="p-2 rounded-full bg-green-500 hover:bg-green-600 text-white transition-colors"
                        title={isPlaying ? 'השהה' : 'נגן'}
                    >
                        {isPlaying ? <Pause size={20} /> : <Play size={20} />}
                    </button>
                    <span className="text-sm font-mono text-gray-700">
                        {formatTime(duration)}
                    </span>
                    <button
                        onClick={uploadRecording}
                        disabled={isUploading}
                        className="p-2 rounded-full bg-blue-500 hover:bg-blue-600 text-white transition-colors disabled:opacity-50"
                        title="שמור הקלטה"
                    >
                        <Upload size={20} />
                    </button>
                    <button
                        onClick={deleteRecording}
                        className="p-2 rounded-full bg-red-500 hover:bg-red-600 text-white transition-colors"
                        title="מחק הקלטה"
                    >
                        <Trash2 size={20} />
                    </button>
                </>
            )}
        </div>
    );
}


