'use client';

import { useState, useRef } from 'react';
import { Upload, Image as ImageIcon, X } from 'lucide-react';

interface FileUploaderProps {
    entityType: 'customer' | 'task' | 'event' | 'project';
    entityId: number;
    onUploadSuccess?: () => void;
    acceptedTypes?: string;
    maxSize?: number; // in MB
}

export default function FileUploader({
    entityType,
    entityId,
    onUploadSuccess,
    acceptedTypes = 'image/*,application/pdf,.doc,.docx,.xls,.xlsx',
    maxSize = 50 // Default 50MB
}: FileUploaderProps) {
    const [selectedFiles, setSelectedFiles] = useState<File[]>([]);
    const [previews, setPreviews] = useState<string[]>([]);
    const [isUploading, setIsUploading] = useState(false);
    const [uploadProgress, setUploadProgress] = useState(0);
    const fileInputRef = useRef<HTMLInputElement>(null);

    const handleFileSelect = (e: React.ChangeEvent<HTMLInputElement>) => {
        const files = Array.from(e.target.files || []);

        // Filter files by size
        const validFiles = files.filter(file => {
            const sizeMB = file.size / (1024 * 1024);
            if (sizeMB > maxSize) {
                alert(`הקובץ ${file.name} גדול מדי. מקסימום: ${maxSize}MB`);
                return false;
            }
            return true;
        });

        setSelectedFiles(validFiles);

        // Create previews for images
        const newPreviews: string[] = [];
        validFiles.forEach(file => {
            if (file.type.startsWith('image/')) {
                const reader = new FileReader();
                reader.onloadend = () => {
                    newPreviews.push(reader.result as string);
                    if (newPreviews.length === validFiles.filter(f => f.type.startsWith('image/')).length) {
                        setPreviews(newPreviews);
                    }
                };
                reader.readAsDataURL(file);
            }
        });
    };

    const removeFile = (index: number) => {
        setSelectedFiles(prev => prev.filter((_, i) => i !== index));
        setPreviews(prev => prev.filter((_, i) => i !== index));
    };

    const uploadFiles = async () => {
        if (selectedFiles.length === 0) return;

        setIsUploading(true);
        let successCount = 0;

        try {
            for (let i = 0; i < selectedFiles.length; i++) {
                const file = selectedFiles[i];
                const formData = new FormData();
                formData.append('file', file);
                formData.append('entity_type', entityType);
                formData.append('entity_id', entityId.toString());

                const response = await fetch('/api/attachments/upload', {
                    method: 'POST',
                    body: formData,
                });

                const data = await response.json();

                if (data.success) {
                    successCount++;
                    setUploadProgress(((i + 1) / selectedFiles.length) * 100);
                } else {
                    console.error('Upload failed for:', file.name);
                }
            }

            if (successCount > 0) {
                alert(`${successCount} קבצים הועלו בהצלחה!`);
                setSelectedFiles([]);
                setPreviews([]);
                setUploadProgress(0);
                onUploadSuccess?.();
                if (fileInputRef.current) {
                    fileInputRef.current.value = '';
                }
            }
        } catch (error) {
            console.error('Error uploading files:', error);
            alert('שגיאה בהעלאת הקבצים');
        } finally {
            setIsUploading(false);
        }
    };

    const triggerFileInput = () => {
        fileInputRef.current?.click();
    };

    return (
        <div className="space-y-3">
            <input
                ref={fileInputRef}
                type="file"
                multiple
                accept={acceptedTypes}
                onChange={handleFileSelect}
                className="hidden"
                aria-label="בחירת קבצים להעלאה"
            />

            <button
                onClick={triggerFileInput}
                className="flex items-center gap-2 px-4 py-2 bg-blue-500 hover:bg-blue-600 text-white rounded-lg transition-colors"
            >
                <ImageIcon size={20} />
                <span>בחר קבצים להעלאה</span>
            </button>

            {selectedFiles.length > 0 && (
                <div className="bg-gray-50 rounded-lg border border-gray-200 p-4">
                    <div className="flex items-center justify-between mb-3">
                        <h4 className="text-sm font-semibold text-gray-700">
                            {selectedFiles.length} קבצים נבחרו
                        </h4>
                        <button
                            onClick={uploadFiles}
                            disabled={isUploading}
                            className="flex items-center gap-2 px-4 py-2 bg-green-500 hover:bg-green-600 text-white rounded-lg transition-colors disabled:opacity-50"
                        >
                            <Upload size={18} />
                            <span>{isUploading ? 'מעלה...' : 'העלה קבצים'}</span>
                        </button>
                    </div>

                    {isUploading && (
                        <div className="mb-3" role="progressbar" aria-label="התקדמות העלאה" aria-valuenow={Math.round(uploadProgress)} aria-valuemin={0} aria-valuemax={100}>
                            <div className="w-full bg-gray-200 rounded-full h-2">
                                <div
                                    className="bg-blue-500 h-2 rounded-full transition-all duration-300"
                                    style={{ width: `${uploadProgress}%` }}
                                />
                            </div>
                            <p className="text-xs text-gray-600 mt-1 text-center">
                                {Math.round(uploadProgress)}%
                            </p>
                        </div>
                    )}

                    <div className="space-y-2">
                        {selectedFiles.map((file, index) => (
                            <div key={index} className="flex items-center gap-3 p-2 bg-white rounded border border-gray-200">
                                {file.type.startsWith('image/') && previews[index] ? (
                                    <img
                                        src={previews[index]}
                                        alt={file.name}
                                        className="w-12 h-12 object-cover rounded"
                                    />
                                ) : (
                                    <div className="w-12 h-12 bg-gray-200 rounded flex items-center justify-center">
                                        <ImageIcon size={24} className="text-gray-400" />
                                    </div>
                                )}
                                <div className="flex-1 min-w-0">
                                    <p className="text-sm font-medium text-gray-700 truncate">
                                        {file.name}
                                    </p>
                                    <p className="text-xs text-gray-500">
                                        {(file.size / 1024).toFixed(2)} KB
                                    </p>
                                </div>
                                <button
                                    onClick={() => removeFile(index)}
                                    className="p-1 text-red-500 hover:bg-red-50 rounded transition-colors"
                                    title="הסר קובץ"
                                    aria-label={`הסר ${file.name}`}
                                >
                                    <X size={18} />
                                </button>
                            </div>
                        ))}
                    </div>
                </div>
            )}
        </div>
    );
}

