'use client';

import { useState, useEffect } from 'react';
import { Download, Trash2, Play, Pause, FileText, Image as ImageIcon, File } from 'lucide-react';
import { Attachment } from '@/types/attachment';

interface AttachmentsGalleryProps {
    entityType: 'customer' | 'task' | 'event' | 'project';
    entityId: number;
    onDelete?: () => void;
}

export default function AttachmentsGallery({ entityType, entityId, onDelete }: AttachmentsGalleryProps) {
    const [attachments, setAttachments] = useState<Attachment[]>([]);
    const [loading, setLoading] = useState(true);
    const [playingAudioId, setPlayingAudioId] = useState<number | null>(null);

    const fetchAttachments = async () => {
        try {
            const response = await fetch(`/api/attachments/${entityType}/${entityId}`);
            const data = await response.json();

            if (data.success) {
                setAttachments(data.attachments);
            }
        } catch (error) {
            console.error('Error fetching attachments:', error);
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchAttachments();
    }, [entityType, entityId]);

    const handleDelete = async (attachmentId: number) => {
        if (!confirm('האם אתה בטוח שברצונך למחוק את הקובץ?')) return;

        try {
            const response = await fetch(`/api/attachments/${entityType}/${entityId}?id=${attachmentId}`, {
                method: 'DELETE',
            });

            const data = await response.json();

            if (data.success) {
                setAttachments(prev => prev.filter(a => a.id !== attachmentId));
                onDelete?.();
            } else {
                alert('שגיאה במחיקת הקובץ');
            }
        } catch (error) {
            console.error('Error deleting attachment:', error);
            alert('שגיאה במחיקת הקובץ');
        }
    };

    const toggleAudioPlayback = (attachmentId: number) => {
        if (playingAudioId === attachmentId) {
            setPlayingAudioId(null);
        } else {
            setPlayingAudioId(attachmentId);
        }
    };

    const formatFileSize = (bytes: number) => {
        if (bytes < 1024) return `${bytes} B`;
        if (bytes < 1024 * 1024) return `${(bytes / 1024).toFixed(2)} KB`;
        return `${(bytes / (1024 * 1024)).toFixed(2)} MB`;
    };

    const formatDuration = (seconds?: number) => {
        if (!seconds) return '';
        const mins = Math.floor(seconds / 60);
        const secs = seconds % 60;
        return `${mins}:${secs.toString().padStart(2, '0')}`;
    };

    const getFileIcon = (attachment: Attachment) => {
        if (attachment.is_voice_recording) {
            return <Play size={24} className="text-purple-500" />;
        }
        if (attachment.mime_type.startsWith('image/')) {
            return <ImageIcon size={24} className="text-blue-500" />;
        }
        if (attachment.mime_type.includes('pdf')) {
            return <FileText size={24} className="text-red-500" />;
        }
        return <File size={24} className="text-gray-500" />;
    };

    if (loading) {
        return (
            <div className="flex items-center justify-center p-8">
                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-500"></div>
            </div>
        );
    }

    if (attachments.length === 0) {
        return (
            <div className="text-center py-8 text-gray-500">
                <File size={48} className="mx-auto mb-2 opacity-50" />
                <p>אין קבצים מצורפים</p>
            </div>
        );
    }

    return (
        <div className="space-y-3">
            <h3 className="text-lg font-semibold text-gray-800">
                קבצים מצורפים ({attachments.length})
            </h3>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                {attachments.map((attachment) => (
                    <div
                        key={attachment.id}
                        className="bg-white border border-gray-200 rounded-lg overflow-hidden hover:shadow-md transition-shadow"
                    >
                        {/* Preview */}
                        <div className="h-40 bg-gray-100 flex items-center justify-center relative">
                            {attachment.mime_type.startsWith('image/') ? (
                                <img
                                    src={attachment.file_path}
                                    alt={attachment.original_name}
                                    className="w-full h-full object-cover"
                                />
                            ) : (
                                <div className="flex flex-col items-center gap-2">
                                    {getFileIcon(attachment)}
                                    {attachment.is_voice_recording && (
                                        <span className="text-sm font-mono text-gray-600">
                                            {formatDuration(attachment.duration)}
                                        </span>
                                    )}
                                </div>
                            )}

                            {attachment.is_voice_recording && (
                                <audio
                                    src={attachment.file_path}
                                    onEnded={() => setPlayingAudioId(null)}
                                    autoPlay={playingAudioId === attachment.id}
                                    className="hidden"
                                />
                            )}
                        </div>

                        {/* Info */}
                        <div className="p-3">
                            <p className="text-sm font-medium text-gray-800 truncate mb-1">
                                {attachment.original_name}
                            </p>
                            <p className="text-xs text-gray-500 mb-2">
                                {formatFileSize(attachment.file_size)} • {new Date(attachment.created_at).toLocaleDateString('he-IL')}
                            </p>

                            {/* Actions */}
                            <div className="flex gap-2">
                                {attachment.is_voice_recording ? (
                                    <button
                                        onClick={() => toggleAudioPlayback(attachment.id)}
                                        className="flex-1 flex items-center justify-center gap-2 px-3 py-2 bg-purple-500 hover:bg-purple-600 text-white rounded transition-colors text-sm"
                                    >
                                        {playingAudioId === attachment.id ? (
                                            <>
                                                <Pause size={16} />
                                                <span>עצור</span>
                                            </>
                                        ) : (
                                            <>
                                                <Play size={16} />
                                                <span>נגן</span>
                                            </>
                                        )}
                                    </button>
                                ) : (
                                    <a
                                        href={attachment.file_path}
                                        download={attachment.original_name}
                                        className="flex-1 flex items-center justify-center gap-2 px-3 py-2 bg-blue-500 hover:bg-blue-600 text-white rounded transition-colors text-sm"
                                    >
                                        <Download size={16} />
                                        <span>הורד</span>
                                    </a>
                                )}

                                <button
                                    onClick={() => handleDelete(attachment.id)}
                                    className="px-3 py-2 bg-red-500 hover:bg-red-600 text-white rounded transition-colors"
                                    title="מחק"
                                >
                                    <Trash2 size={16} />
                                </button>
                            </div>
                        </div>
                    </div>
                ))}
            </div>
        </div>
    );
}


