import sqlite3 from 'sqlite3';
import path from 'path';

const phonesDbPath = path.join(process.cwd(), 'phones.db');
const comprehensiveDbPath = path.join(process.cwd(), 'comprehensive.db');

const phonesDb = new sqlite3.Database(phonesDbPath);
const comprehensiveDb = new sqlite3.Database(comprehensiveDbPath);

async function syncCustomersFromDomains() {
    console.log('🔄 Starting customer sync from domains...\n');

    // Get all unique customers from phones table
    phonesDb.all('SELECT DISTINCT owner, email, phone FROM phones ORDER BY owner', async (err, rows: any[]) => {
        if (err) {
            console.error('❌ Error fetching phones:', err);
            process.exit(1);
        }

        console.log(`📊 Found ${rows.length} unique customers in domains\n`);

        let created = 0;
        let existing = 0;
        let errors = 0;

        for (const row of rows) {
            try {
                // Check if customer exists
                const customerExists = await new Promise<any>((resolve) => {
                    comprehensiveDb.get(
                        'SELECT * FROM customers WHERE email = ?',
                        [row.email],
                        (err, customer) => {
                            if (err) {
                                console.error('Error checking customer:', err);
                                resolve(null);
                            } else {
                                resolve(customer);
                            }
                        }
                    );
                });

                if (customerExists) {
                    console.log(`✅ Customer exists: ${row.owner} (${row.email})`);
                    existing++;
                } else {
                    // Create new customer
                    const customerId = `customer_${Date.now()}_${Math.random().toString(36).substring(7)}`;
                    const now = new Date().toISOString();

                    await new Promise<void>((resolve, reject) => {
                        comprehensiveDb.run(
                            `INSERT INTO customers (id, name, email, phone, company, status, stage, value, created_date, last_activity, updated_at)
                             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
                            [
                                customerId,
                                row.owner,
                                row.email,
                                row.phone,
                                '', // company
                                'active', // status
                                'quote', // stage
                                0, // value
                                now,
                                now,
                                now
                            ],
                            (err) => {
                                if (err) {
                                    reject(err);
                                } else {
                                    resolve();
                                }
                            }
                        );
                    });

                    console.log(`✅ Created customer: ${row.owner} (${row.email}) → ${customerId}`);
                    created++;
                }

                // Small delay to avoid ID conflicts
                await new Promise(resolve => setTimeout(resolve, 10));

            } catch (error) {
                console.error(`❌ Error processing ${row.owner}:`, error);
                errors++;
            }
        }

        console.log(`\n📊 Sync Summary:`);
        console.log(`   ✅ Created: ${created}`);
        console.log(`   ℹ️  Existing: ${existing}`);
        console.log(`   ❌ Errors: ${errors}`);
        console.log(`   📈 Total: ${rows.length}\n`);

        phonesDb.close();
        comprehensiveDb.close();

        console.log('✅ Sync completed successfully!');
        process.exit(0);
    });
}

syncCustomersFromDomains();









