const sqlite3 = require('sqlite3');
const path = require('path');

const seedComprehensiveDB = async () => {
    return new Promise((resolve, reject) => {
        const dbPath = path.join(process.cwd(), 'comprehensive.db');
        const db = new sqlite3.Database(dbPath, (err) => {
            if (err) {
                console.error('Error opening comprehensive.db:', err);
                reject(err);
                return;
            }
            console.log('Connected to comprehensive.db for seeding...');
        });

        // Sample customers
        const sampleCustomers = [
            {
                id: 'customer_001',
                name: 'חברת טכנולוגיה בע״מ',
                email: 'info@tech-company.co.il',
                phone: '03-1234567',
                company: 'חברת טכנולוגיה בע״מ',
                status: 'active',
                stage: 'client',
                assigned_sales_rep: 'טיטי',
                value: 15000,
                notes: 'לקוח ותיק ומרוצה',
                created_date: new Date().toISOString(),
                last_activity: new Date().toISOString(),
                updated_at: new Date().toISOString()
            },
            {
                id: 'customer_002',
                name: 'עסק מקומי',
                email: 'contact@local-business.co.il',
                phone: '04-7654321',
                company: 'עסק מקומי',
                status: 'lead',
                stage: 'quote',
                assigned_sales_rep: 'טיטי',
                value: 5000,
                notes: 'פוטנציאל גבוה',
                created_date: new Date().toISOString(),
                last_activity: new Date().toISOString(),
                updated_at: new Date().toISOString()
            },
            {
                id: 'customer_003',
                name: 'סטארט-אפ חדש',
                email: 'hello@startup.co.il',
                phone: '050-1234567',
                company: 'סטארט-אפ חדש',
                status: 'prospect',
                stage: 'negotiation',
                assigned_sales_rep: 'טיטי',
                value: 8000,
                notes: 'בהתחלה של תהליך המכירה',
                created_date: new Date().toISOString(),
                last_activity: new Date().toISOString(),
                updated_at: new Date().toISOString()
            }
        ];

        // Sample domains
        const sampleDomains = [
            {
                id: 'domain_001',
                name: 'tech-company.co.il',
                customer_id: 'customer_001',
                status: 'active',
                registration_date: '2023-01-15',
                expiry_date: '2025-01-15',
                provider: 'רשם דומיינים',
                price: 120,
                notes: 'דומיין ראשי של החברה',
                created_at: new Date().toISOString(),
                updated_at: new Date().toISOString()
            },
            {
                id: 'domain_002',
                name: 'local-business.co.il',
                customer_id: 'customer_002',
                status: 'active',
                registration_date: '2023-06-20',
                expiry_date: '2025-06-20',
                provider: 'רשם דומיינים',
                price: 120,
                notes: 'דומיין עסקי',
                created_at: new Date().toISOString(),
                updated_at: new Date().toISOString()
            },
            {
                id: 'domain_003',
                name: 'startup.co.il',
                customer_id: 'customer_003',
                status: 'pending',
                registration_date: null,
                expiry_date: null,
                provider: null,
                price: 120,
                notes: 'בהמתנה לאישור הלקוח',
                created_at: new Date().toISOString(),
                updated_at: new Date().toISOString()
            }
        ];

        // Sample projects
        const sampleProjects = [
            {
                id: 'project_001',
                name: 'אתר אינטרנט - חברת טכנולוגיה',
                customer_id: 'customer_001',
                status: 'active',
                start_date: '2023-02-01',
                end_date: '2023-08-01',
                budget: 15000,
                description: 'פיתוח אתר אינטרנט מלא עם מערכת ניהול תוכן',
                created_at: new Date().toISOString(),
                updated_at: new Date().toISOString()
            },
            {
                id: 'project_002',
                name: 'אתר תדמית - עסק מקומי',
                customer_id: 'customer_002',
                status: 'quote',
                start_date: null,
                end_date: null,
                budget: 5000,
                description: 'אתר תדמית פשוט לעסק מקומי',
                created_at: new Date().toISOString(),
                updated_at: new Date().toISOString()
            }
        ];

        db.serialize(() => {
            // Insert customers
            sampleCustomers.forEach((customer, index) => {
                db.run(`
                    INSERT OR REPLACE INTO customers (
                        id, name, email, phone, company, status, stage, 
                        assigned_sales_rep, value, notes, created_date, last_activity, updated_at
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                `, [
                    customer.id,
                    customer.name,
                    customer.email,
                    customer.phone,
                    customer.company,
                    customer.status,
                    customer.stage,
                    customer.assigned_sales_rep,
                    customer.value,
                    customer.notes,
                    customer.created_date,
                    customer.last_activity,
                    customer.updated_at
                ], (err) => {
                    if (err) {
                        console.error(`Error inserting customer ${index + 1}:`, err);
                    } else {
                        console.log(`✅ Customer ${index + 1} inserted: ${customer.name}`);
                    }
                });
            });

            // Insert domains
            sampleDomains.forEach((domain, index) => {
                db.run(`
                    INSERT OR REPLACE INTO domains (
                        id, name, customer_id, status, registration_date, expiry_date,
                        provider, price, notes, created_at, updated_at
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                `, [
                    domain.id,
                    domain.name,
                    domain.customer_id,
                    domain.status,
                    domain.registration_date,
                    domain.expiry_date,
                    domain.provider,
                    domain.price,
                    domain.notes,
                    domain.created_at,
                    domain.updated_at
                ], (err) => {
                    if (err) {
                        console.error(`Error inserting domain ${index + 1}:`, err);
                    } else {
                        console.log(`✅ Domain ${index + 1} inserted: ${domain.name}`);
                    }
                });
            });

            // Insert projects
            sampleProjects.forEach((project, index) => {
                db.run(`
                    INSERT OR REPLACE INTO projects (
                        id, name, customer_id, status, start_date, end_date,
                        budget, description, created_at, updated_at
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                `, [
                    project.id,
                    project.name,
                    project.customer_id,
                    project.status,
                    project.start_date,
                    project.end_date,
                    project.budget,
                    project.description,
                    project.created_at,
                    project.updated_at
                ], (err) => {
                    if (err) {
                        console.error(`Error inserting project ${index + 1}:`, err);
                    } else {
                        console.log(`✅ Project ${index + 1} inserted: ${project.name}`);
                    }
                });
            });
        });

        db.close((err) => {
            if (err) {
                console.error('Error closing comprehensive.db:', err);
                reject(err);
            } else {
                console.log('✅ Sample data seeded successfully!');
                resolve();
            }
        });
    });
};

const main = async () => {
    try {
        console.log('Seeding sample data...');
        await seedComprehensiveDB();
        console.log('🎉 All done! You should now see customers, domains, and projects in your app.');
        process.exit(0);
    } catch (error) {
        console.error('❌ Error seeding data:', error);
        process.exit(1);
    }
};

main();
