const sqlite3 = require('sqlite3').verbose();
const path = require('path');

const dbPath = path.join(__dirname, '../comprehensive.db');
const db = new sqlite3.Database(dbPath, (err) => {
    if (err) {
        console.error('❌ Error opening database:', err.message);
        process.exit(1);
    }
    console.log('✅ Connected to comprehensive.db');
});

try {
    // Get first customer for demo event
    db.get('SELECT id, name FROM customers LIMIT 1', (err, customer) => {
        if (err) {
            console.error('❌ Error fetching customer:', err.message);
            db.close();
            process.exit(1);
        }

        if (!customer) {
            console.log('⚠️  No customers found. Please import customers first.');
            db.close();
            process.exit(0);
        }

        console.log('✅ Found customer:', customer.name);

        // Insert sample events
        const events = [
            {
                title: 'פגישת היכרות עם לקוח',
                description: 'פגישה ראשונה לדיון על פרויקט חדש',
                start_datetime: new Date(Date.now() + 86400000).toISOString(), // Tomorrow
                end_datetime: new Date(Date.now() + 90000000).toISOString(), // Tomorrow + 1 hour
                location: 'משרד הלקוח',
                customer_id: customer.id,
                event_type: 'meeting',
                status: 'scheduled',
                attendees: JSON.stringify(['מנהל פרויקט', 'נציג לקוח']),
                reminder_minutes: 30
            },
            {
                title: 'שיחת טלפון - מעקב',
                description: 'שיחת מעקב אחרי הצעת המחיר',
                start_datetime: new Date(Date.now() + 172800000).toISOString(), // Day after tomorrow
                end_datetime: null,
                location: null,
                customer_id: customer.id,
                event_type: 'call',
                status: 'scheduled',
                attendees: null,
                reminder_minutes: 15
            },
            {
                title: 'מצגת פרויקט',
                description: 'הצגת התוכנית למימוש הפרויקט',
                start_datetime: new Date(Date.now() + 259200000).toISOString(), // In 3 days
                end_datetime: new Date(Date.now() + 266400000).toISOString(), // In 3 days + 2 hours
                location: 'zoom',
                customer_id: customer.id,
                event_type: 'presentation',
                status: 'scheduled',
                attendees: JSON.stringify(['צוות פיתוח', 'מנהל פרויקט', 'לקוח']),
                reminder_minutes: 60
            }
        ];

        const stmt = db.prepare(`
            INSERT INTO events (
                title, description, start_datetime, end_datetime, location,
                customer_id, event_type, status, attendees, reminder_minutes,
                created_at, updated_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, datetime('now'), datetime('now'))
        `);

        events.forEach(event => {
            stmt.run(
                event.title,
                event.description,
                event.start_datetime,
                event.end_datetime,
                event.location,
                event.customer_id,
                event.event_type,
                event.status,
                event.attendees,
                event.reminder_minutes,
                (err) => {
                    if (err) {
                        console.error('❌ Error inserting event:', err.message);
                    } else {
                        console.log('✅ Inserted event:', event.title);
                    }
                }
            );
        });

        stmt.finalize((err) => {
            if (err) {
                console.error('❌ Error finalizing statement:', err.message);
            } else {
                console.log('');
                console.log('🎉 Successfully seeded events!');
            }
            db.close();
        });
    });

} catch (error) {
    console.error('❌ Error:', error);
    db.close();
    process.exit(1);
}

