import { getCrmDatabase, initCrmDatabase, clearCrmTables } from '../src/lib/crmDatabase';

const sampleCustomers = [
    {
        id: 'customer_1',
        name: 'משה כהן',
        email: 'moshe@example.com',
        phone: '050-1234567',
        company: 'חברת כהן בע"מ',
        status: 'active',
        stage: 'project',
        assigned_sales_rep: 'שרה לוי',
        value: 50000,
        notes: 'לקוח VIP - פרויקט גדול',
        created_date: '2024-01-15T10:00:00.000Z',
        last_activity: '2024-12-20T14:30:00.000Z',
        updated_at: '2024-12-20T14:30:00.000Z'
    },
    {
        id: 'customer_2',
        name: 'רחל גולד',
        email: 'rachel@gold.com',
        phone: '052-9876543',
        company: 'גולד טכנולוגיות',
        status: 'active',
        stage: 'quote',
        assigned_sales_rep: 'דוד ישראלי',
        value: 25000,
        notes: 'מחכה להצעת מחיר',
        created_date: '2024-02-10T09:15:00.000Z',
        last_activity: '2024-12-19T16:45:00.000Z',
        updated_at: '2024-12-19T16:45:00.000Z'
    },
    {
        id: 'customer_3',
        name: 'יוסי לוי',
        email: 'yossi@levi.co.il',
        phone: '054-5555555',
        company: 'לוי ושות\'',
        status: 'lead',
        stage: 'quote',
        assigned_sales_rep: 'שרה לוי',
        value: 15000,
        notes: 'ליד חדש - צריך ליצור קשר',
        created_date: '2024-12-18T11:20:00.000Z',
        last_activity: '2024-12-18T11:20:00.000Z',
        updated_at: '2024-12-18T11:20:00.000Z'
    },
    {
        id: 'customer_4',
        name: 'מיכל רוזן',
        email: 'michal@rosen.org',
        phone: '053-1111111',
        company: 'רוזן קונסלטינג',
        status: 'inactive',
        stage: 'closed',
        assigned_sales_rep: 'דוד ישראלי',
        value: 75000,
        notes: 'פרויקט הושלם בהצלחה',
        created_date: '2024-03-05T08:30:00.000Z',
        last_activity: '2024-11-30T17:00:00.000Z',
        updated_at: '2024-11-30T17:00:00.000Z'
    },
    {
        id: 'customer_5',
        name: 'אבי שטרן',
        email: 'avi@stern.net',
        phone: '050-9999999',
        company: 'שטרן פתרונות',
        status: 'active',
        stage: 'follow_up',
        assigned_sales_rep: 'שרה לוי',
        value: 30000,
        notes: 'צריך מעקב - תשלום מאוחר',
        created_date: '2024-04-12T13:45:00.000Z',
        last_activity: '2024-12-15T10:15:00.000Z',
        updated_at: '2024-12-15T10:15:00.000Z'
    }
];

async function seedCrmData() {
    try {
        console.log('🌱 Starting CRM data seeding...');

        // Initialize database
        await initCrmDatabase();

        // Clear existing data
        await clearCrmTables();
        console.log('✅ Cleared existing CRM data');

        // Insert sample customers
        const database = getCrmDatabase();
        const insertCustomer = database.prepare(`
      INSERT INTO customers (
        id, name, email, phone, company, status, stage, 
        assigned_sales_rep, value, notes, created_date, last_activity, updated_at
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    `);

        for (const customer of sampleCustomers) {
            insertCustomer.run(
                customer.id,
                customer.name,
                customer.email,
                customer.phone,
                customer.company,
                customer.status,
                customer.stage,
                customer.assigned_sales_rep,
                customer.value,
                customer.notes,
                customer.created_date,
                customer.last_activity,
                customer.updated_at
            );
        }

        console.log(`✅ Inserted ${sampleCustomers.length} customers`);

        // Insert sample quotes
        const sampleQuotes = [
            {
                id: 'quote_1',
                customer_id: 'customer_1',
                title: 'פרויקט פיתוח אתר',
                amount: 50000,
                status: 'approved',
                created_date: '2024-01-15T10:00:00.000Z',
                valid_until: '2024-02-15T10:00:00.000Z',
                notes: 'אושר על ידי הלקוח'
            },
            {
                id: 'quote_2',
                customer_id: 'customer_2',
                title: 'מערכת CRM',
                amount: 25000,
                status: 'sent',
                created_date: '2024-02-10T09:15:00.000Z',
                valid_until: '2024-03-10T09:15:00.000Z',
                notes: 'נשלח ללקוח - ממתין לתגובה'
            },
            {
                id: 'quote_3',
                customer_id: 'customer_3',
                title: 'ייעוץ טכנולוגי',
                amount: 15000,
                status: 'pending',
                created_date: '2024-12-18T11:20:00.000Z',
                valid_until: '2025-01-18T11:20:00.000Z',
                notes: 'בהכנה'
            }
        ];

        const insertQuote = database.prepare(`
      INSERT INTO quotes (id, customer_id, title, amount, status, created_date, valid_until, notes)
      VALUES (?, ?, ?, ?, ?, ?, ?, ?)
    `);

        for (const quote of sampleQuotes) {
            insertQuote.run(
                quote.id,
                quote.customer_id,
                quote.title,
                quote.amount,
                quote.status,
                quote.created_date,
                quote.valid_until,
                quote.notes
            );
        }

        console.log(`✅ Inserted ${sampleQuotes.length} quotes`);

        // Insert sample projects
        const sampleProjects = [
            {
                id: 'project_1',
                customer_id: 'customer_1',
                title: 'פיתוח אתר איקומרס',
                description: 'אתר מכירות מקוון עם מערכת תשלומים',
                status: 'in_progress',
                start_date: '2024-01-20',
                end_date: '2024-06-20',
                budget: 50000,
                progress: 75,
                created_date: '2024-01-20T08:00:00.000Z',
                updated_at: '2024-12-20T14:30:00.000Z'
            },
            {
                id: 'project_2',
                customer_id: 'customer_4',
                title: 'מערכת ניהול מלאי',
                description: 'מערכת לניהול מלאי וזמנים',
                status: 'completed',
                start_date: '2024-03-10',
                end_date: '2024-08-10',
                budget: 75000,
                progress: 100,
                created_date: '2024-03-10T09:00:00.000Z',
                updated_at: '2024-11-30T17:00:00.000Z'
            }
        ];

        const insertProject = database.prepare(`
      INSERT INTO projects (id, customer_id, title, description, status, start_date, end_date, budget, progress, created_date, updated_at)
      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    `);

        for (const project of sampleProjects) {
            insertProject.run(
                project.id,
                project.customer_id,
                project.title,
                project.description,
                project.status,
                project.start_date,
                project.end_date,
                project.budget,
                project.progress,
                project.created_date,
                project.updated_at
            );
        }

        console.log(`✅ Inserted ${sampleProjects.length} projects`);
        console.log('🎉 CRM data seeding completed successfully!');

    } catch (error) {
        console.error('❌ Error seeding CRM data:', error);
        process.exit(1);
    }
}

// Run the seeding
seedCrmData();
