const Database = require('better-sqlite3');
const path = require('path');

let db = null;

function getComprehensiveDatabase() {
    if (!db) {
        const dbPath = path.join(process.cwd(), 'comprehensive.db');
        db = new Database(dbPath);
        console.log(`Connected to Comprehensive SQLite database at: ${dbPath}`);
    }
    return db;
}

async function initComprehensiveDatabase() {
    const database = getComprehensiveDatabase();

    // Enable foreign key constraints
    database.pragma('foreign_keys = ON');

    // Create all tables with proper relationships

    // 1. CUSTOMERS table (main entity)
    database.exec(`
        CREATE TABLE IF NOT EXISTS customers (
            id TEXT PRIMARY KEY,
            name TEXT NOT NULL,
            email TEXT UNIQUE,
            phone TEXT,
            company TEXT,
            status TEXT CHECK(status IN ('active', 'inactive', 'lead')) DEFAULT 'lead',
            stage TEXT CHECK(stage IN ('quote', 'project', 'follow_up', 'closed')) DEFAULT 'quote',
            assigned_sales_rep TEXT,
            value REAL DEFAULT 0,
            notes TEXT,
            created_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            last_activity DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
        )
    `);
    console.log('Customers table created successfully or already exists');

    // 2. DOMAINS table (linked to customers)
    database.exec(`
        CREATE TABLE IF NOT EXISTS domains (
            id TEXT PRIMARY KEY,
            customer_id TEXT NOT NULL,
            domain_name TEXT NOT NULL,
            website TEXT,
            owner TEXT,
            email TEXT,
            package TEXT,
            amount REAL,
            status TEXT CHECK(status IN ('active', 'inactive', 'pending', 'expired')) DEFAULT 'active',
            priority TEXT CHECK(priority IN ('low', 'medium', 'high')) DEFAULT 'medium',
            next_payment TEXT,
            last_update TEXT,
            notes TEXT,
            created_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (customer_id) REFERENCES customers (id) ON DELETE CASCADE
        )
    `);
    console.log('Domains table created successfully or already exists');

    // 3. PROJECTS table (linked to customers)
    database.exec(`
        CREATE TABLE IF NOT EXISTS projects (
            id TEXT PRIMARY KEY,
            customer_id TEXT NOT NULL,
            title TEXT NOT NULL,
            description TEXT,
            status TEXT CHECK(status IN ('planning', 'in_progress', 'completed', 'on_hold', 'cancelled')) DEFAULT 'planning',
            start_date DATE,
            end_date DATE,
            budget REAL,
            progress INTEGER DEFAULT 0,
            assigned_team TEXT,
            notes TEXT,
            created_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (customer_id) REFERENCES customers (id) ON DELETE CASCADE
        )
    `);
    console.log('Projects table created successfully or already exists');

    // 4. QUOTES table (linked to customers)
    database.exec(`
        CREATE TABLE IF NOT EXISTS quotes (
            id TEXT PRIMARY KEY,
            customer_id TEXT NOT NULL,
            title TEXT NOT NULL,
            amount REAL NOT NULL,
            status TEXT CHECK(status IN ('pending', 'sent', 'approved', 'rejected', 'expired')) DEFAULT 'pending',
            valid_until DATETIME,
            description TEXT,
            terms TEXT,
            notes TEXT,
            created_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (customer_id) REFERENCES customers (id) ON DELETE CASCADE
        )
    `);
    console.log('Quotes table created successfully or already exists');

    // 5. PAYMENTS table (linked to customers, domains, projects, quotes)
    database.exec(`
        CREATE TABLE IF NOT EXISTS payments (
            id TEXT PRIMARY KEY,
            customer_id TEXT NOT NULL,
            domain_id TEXT,
            project_id TEXT,
            quote_id TEXT,
            amount REAL NOT NULL,
            payment_type TEXT CHECK(payment_type IN ('domain_renewal', 'project_payment', 'quote_payment', 'service_payment', 'other')) NOT NULL,
            status TEXT CHECK(status IN ('pending', 'completed', 'failed', 'cancelled', 'refunded')) DEFAULT 'pending',
            payment_method TEXT CHECK(payment_method IN ('credit_card', 'bank_transfer', 'paypal', 'cash', 'check', 'other')) DEFAULT 'credit_card',
            payment_date DATETIME,
            due_date DATETIME,
            invoice_number TEXT,
            description TEXT,
            notes TEXT,
            created_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (customer_id) REFERENCES customers (id) ON DELETE CASCADE,
            FOREIGN KEY (domain_id) REFERENCES domains (id) ON DELETE SET NULL,
            FOREIGN KEY (project_id) REFERENCES projects (id) ON DELETE SET NULL,
            FOREIGN KEY (quote_id) REFERENCES quotes (id) ON DELETE SET NULL
        )
    `);
    console.log('Payments table created successfully or already exists');

    // 6. ACTIVITIES table (linked to customers)
    database.exec(`
        CREATE TABLE IF NOT EXISTS activities (
            id TEXT PRIMARY KEY,
            customer_id TEXT NOT NULL,
            activity_type TEXT CHECK(activity_type IN ('call', 'email', 'meeting', 'note', 'task', 'reminder', 'payment', 'other')) NOT NULL,
            title TEXT NOT NULL,
            description TEXT,
            status TEXT CHECK(status IN ('pending', 'completed', 'cancelled')) DEFAULT 'pending',
            priority TEXT CHECK(priority IN ('low', 'medium', 'high')) DEFAULT 'medium',
            due_date DATETIME,
            completed_date DATETIME,
            assigned_to TEXT,
            duration_minutes INTEGER,
            outcome TEXT,
            notes TEXT,
            created_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (customer_id) REFERENCES customers (id) ON DELETE CASCADE
        )
    `);
    console.log('Activities table created successfully or already exists');

    // 7. REMINDERS table (linked to customers, domains, projects)
    database.exec(`
        CREATE TABLE IF NOT EXISTS reminders (
            id TEXT PRIMARY KEY,
            customer_id TEXT NOT NULL,
            domain_id TEXT,
            project_id TEXT,
            reminder_type TEXT CHECK(reminder_type IN ('domain_renewal', 'payment_due', 'project_deadline', 'follow_up', 'meeting', 'other')) NOT NULL,
            title TEXT NOT NULL,
            description TEXT,
            reminder_date DATETIME NOT NULL,
            status TEXT CHECK(status IN ('pending', 'completed', 'cancelled', 'overdue')) DEFAULT 'pending',
            priority TEXT CHECK(priority IN ('low', 'medium', 'high')) DEFAULT 'medium',
            recurring TEXT CHECK(recurring IN ('none', 'daily', 'weekly', 'monthly', 'yearly')) DEFAULT 'none',
            completed_date DATETIME,
            notes TEXT,
            created_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (customer_id) REFERENCES customers (id) ON DELETE CASCADE,
            FOREIGN KEY (domain_id) REFERENCES domains (id) ON DELETE SET NULL,
            FOREIGN KEY (project_id) REFERENCES projects (id) ON DELETE SET NULL
        )
    `);
    console.log('Reminders table created successfully or already exists');

    // 8. EMAIL_SERVICES table (linked to domains)
    database.exec(`
        CREATE TABLE IF NOT EXISTS email_services (
            id TEXT PRIMARY KEY,
            domain_id TEXT NOT NULL,
            service_type TEXT CHECK(service_type IN ('hosting', 'email_accounts', 'ssl_certificate', 'backup', 'security', 'other')) NOT NULL,
            service_name TEXT NOT NULL,
            provider TEXT,
            status TEXT CHECK(status IN ('active', 'inactive', 'pending', 'suspended')) DEFAULT 'active',
            start_date DATE,
            end_date DATE,
            monthly_cost REAL,
            features TEXT,
            notes TEXT,
            created_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (domain_id) REFERENCES domains (id) ON DELETE CASCADE
        )
    `);
    console.log('Email services table created successfully or already exists');

    // 9. TECHNICAL_SERVICES table (linked to domains)
    database.exec(`
        CREATE TABLE IF NOT EXISTS technical_services (
            id TEXT PRIMARY KEY,
            domain_id TEXT NOT NULL,
            service_type TEXT CHECK(service_type IN ('dns_management', 'ssl_certificate', 'cdn', 'monitoring', 'backup', 'security', 'performance', 'other')) NOT NULL,
            service_name TEXT NOT NULL,
            provider TEXT,
            status TEXT CHECK(status IN ('active', 'inactive', 'pending', 'maintenance')) DEFAULT 'active',
            start_date DATE,
            end_date DATE,
            monthly_cost REAL,
            configuration TEXT,
            notes TEXT,
            created_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (domain_id) REFERENCES domains (id) ON DELETE CASCADE
        )
    `);
    console.log('Technical services table created successfully or already exists');

    // Create indexes for better performance
    database.exec(`
        CREATE INDEX IF NOT EXISTS idx_customers_email ON customers(email);
        CREATE INDEX IF NOT EXISTS idx_customers_status ON customers(status);
        CREATE INDEX IF NOT EXISTS idx_domains_customer_id ON domains(customer_id);
        CREATE INDEX IF NOT EXISTS idx_domains_status ON domains(status);
        CREATE INDEX IF NOT EXISTS idx_projects_customer_id ON projects(customer_id);
        CREATE INDEX IF NOT EXISTS idx_projects_status ON projects(status);
        CREATE INDEX IF NOT EXISTS idx_quotes_customer_id ON quotes(customer_id);
        CREATE INDEX IF NOT EXISTS idx_quotes_status ON quotes(status);
        CREATE INDEX IF NOT EXISTS idx_payments_customer_id ON payments(customer_id);
        CREATE INDEX IF NOT EXISTS idx_payments_status ON payments(status);
        CREATE INDEX IF NOT EXISTS idx_activities_customer_id ON activities(customer_id);
        CREATE INDEX IF NOT EXISTS idx_reminders_customer_id ON reminders(customer_id);
        CREATE INDEX IF NOT EXISTS idx_reminders_reminder_date ON reminders(reminder_date);
        CREATE INDEX IF NOT EXISTS idx_email_services_domain_id ON email_services(domain_id);
        CREATE INDEX IF NOT EXISTS idx_technical_services_domain_id ON technical_services(domain_id);
    `);
    console.log('Database indexes created successfully');
}

async function seedComprehensiveData() {
    try {
        await initComprehensiveDatabase();
        const db = getComprehensiveDatabase();

        console.log('🌱 Starting comprehensive data seeding...');

        // Clear existing data
        console.log('🧹 Clearing existing data...');
        db.exec(`
            DELETE FROM technical_services;
            DELETE FROM email_services;
            DELETE FROM reminders;
            DELETE FROM activities;
            DELETE FROM payments;
            DELETE FROM quotes;
            DELETE FROM projects;
            DELETE FROM domains;
            DELETE FROM customers;
        `);

        // Insert sample customers
        console.log('👥 Creating sample customers...');
        const customers = [
            {
                id: 'customer_1',
                name: 'חברת טכנולוגיות מתקדמות',
                email: 'info@tech-advanced.co.il',
                phone: '050-1234567',
                company: 'טכנולוגיות מתקדמות בע"מ',
                status: 'active',
                stage: 'project',
                assigned_sales_rep: 'דוד כהן',
                value: 50000,
                notes: 'לקוח VIP - פרויקט גדול בפיתוח'
            },
            {
                id: 'customer_2',
                name: 'חנות אונליין',
                email: 'contact@shop-online.co.il',
                phone: '052-9876543',
                company: 'חנות אונליין בע"מ',
                status: 'active',
                stage: 'follow_up',
                assigned_sales_rep: 'שרה לוי',
                value: 15000,
                notes: 'דורש מעקב שוטף'
            },
            {
                id: 'customer_3',
                name: 'משרד עורכי דין',
                email: 'office@law-firm.co.il',
                phone: '03-1234567',
                company: 'משרד עורכי דין כהן ושות\'',
                status: 'lead',
                stage: 'quote',
                assigned_sales_rep: 'מיכל רוזן',
                value: 8000,
                notes: 'ליד חדש - ממתין להצעת מחיר'
            },
            {
                id: 'customer_4',
                name: 'מסעדה מקומית',
                email: 'info@local-restaurant.co.il',
                phone: '054-5555555',
                company: 'מסעדת השף המקומי',
                status: 'active',
                stage: 'project',
                assigned_sales_rep: 'יוסי גולד',
                value: 25000,
                notes: 'פיתוח אתר מסעדה עם מערכת הזמנות'
            }
        ];

        customers.forEach(customer => {
            db.prepare(`
                INSERT INTO customers (
                    id, name, email, phone, company, status, stage, 
                    assigned_sales_rep, value, notes, created_date, last_activity, updated_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            `).run(
                customer.id,
                customer.name,
                customer.email,
                customer.phone,
                customer.company,
                customer.status,
                customer.stage,
                customer.assigned_sales_rep,
                customer.value,
                customer.notes,
                new Date().toISOString(),
                new Date().toISOString(),
                new Date().toISOString()
            );
        });

        // Insert sample domains
        console.log('🌐 Creating sample domains...');
        const domains = [
            {
                id: 'domain_1',
                customer_id: 'customer_1',
                domain_name: 'tech-advanced.co.il',
                website: 'https://tech-advanced.co.il',
                owner: 'חברת טכנולוגיות מתקדמות',
                email: 'info@tech-advanced.co.il',
                package: 'Business Pro',
                amount: 299,
                status: 'active',
                priority: 'high',
                next_payment: '2024-12-15',
                notes: 'דומיין ראשי - חידוש אוטומטי'
            },
            {
                id: 'domain_2',
                customer_id: 'customer_1',
                domain_name: 'api.tech-advanced.co.il',
                website: 'https://api.tech-advanced.co.il',
                owner: 'חברת טכנולוגיות מתקדמות',
                email: 'dev@tech-advanced.co.il',
                package: 'Developer',
                amount: 199,
                status: 'active',
                priority: 'medium',
                next_payment: '2024-11-20',
                notes: 'דומיין API'
            },
            {
                id: 'domain_3',
                customer_id: 'customer_2',
                domain_name: 'shop-online.co.il',
                website: 'https://shop-online.co.il',
                owner: 'חנות אונליין',
                email: 'admin@shop-online.co.il',
                package: 'E-commerce',
                amount: 399,
                status: 'active',
                priority: 'high',
                next_payment: '2025-01-10',
                notes: 'דומיין חנות אונליין'
            },
            {
                id: 'domain_4',
                customer_id: 'customer_3',
                domain_name: 'law-firm.co.il',
                website: 'https://law-firm.co.il',
                owner: 'משרד עורכי דין',
                email: 'office@law-firm.co.il',
                package: 'Professional',
                amount: 249,
                status: 'pending',
                priority: 'medium',
                next_payment: '2024-12-01',
                notes: 'דומיין חדש - ממתין להפעלה'
            }
        ];

        domains.forEach(domain => {
            db.prepare(`
                INSERT INTO domains (
                    id, customer_id, domain_name, website, owner, email, package,
                    amount, status, priority, next_payment, notes, created_date, updated_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            `).run(
                domain.id,
                domain.customer_id,
                domain.domain_name,
                domain.website,
                domain.owner,
                domain.email,
                domain.package,
                domain.amount,
                domain.status,
                domain.priority,
                domain.next_payment,
                domain.notes,
                new Date().toISOString(),
                new Date().toISOString()
            );
        });

        // Insert sample projects
        console.log('📋 Creating sample projects...');
        const projects = [
            {
                id: 'project_1',
                customer_id: 'customer_1',
                title: 'פיתוח מערכת CRM מתקדמת',
                description: 'פיתוח מערכת CRM מלאה עם דשבורד, ניהול לקוחות ופרויקטים',
                status: 'in_progress',
                start_date: '2024-09-01',
                end_date: '2024-12-31',
                budget: 50000,
                progress: 65,
                assigned_team: 'צוות פיתוח A',
                notes: 'פרויקט גדול - מתקדם לפי לוח הזמנים'
            },
            {
                id: 'project_2',
                customer_id: 'customer_2',
                title: 'עיצוב אתר חנות אונליין',
                description: 'עיצוב ופיתוח אתר חנות אונליין עם מערכת תשלומים',
                status: 'completed',
                start_date: '2024-08-01',
                end_date: '2024-10-15',
                budget: 15000,
                progress: 100,
                assigned_team: 'צוות עיצוב',
                notes: 'הושלם בהצלחה - לקוח מרוצה'
            },
            {
                id: 'project_3',
                customer_id: 'customer_4',
                title: 'אתר מסעדה עם מערכת הזמנות',
                description: 'פיתוח אתר מסעדה עם מערכת הזמנות אונליין ומשלוחים',
                status: 'planning',
                start_date: '2024-11-01',
                end_date: '2025-02-28',
                budget: 25000,
                progress: 10,
                assigned_team: 'צוות פיתוח B',
                notes: 'בשלב תכנון - ממתין לאישור לקוח'
            }
        ];

        projects.forEach(project => {
            db.prepare(`
                INSERT INTO projects (
                    id, customer_id, title, description, status, start_date, end_date,
                    budget, progress, assigned_team, notes, created_date, updated_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            `).run(
                project.id,
                project.customer_id,
                project.title,
                project.description,
                project.status,
                project.start_date,
                project.end_date,
                project.budget,
                project.progress,
                project.assigned_team,
                project.notes,
                new Date().toISOString(),
                new Date().toISOString()
            );
        });

        // Insert sample quotes
        console.log('💰 Creating sample quotes...');
        const quotes = [
            {
                id: 'quote_1',
                customer_id: 'customer_3',
                title: 'הצעת מחיר לפיתוח אתר משרד עורכי דין',
                amount: 8000,
                status: 'sent',
                valid_until: '2024-10-31',
                description: 'פיתוח אתר מקצועי למשרד עורכי דין עם מערכת ניהול תיקים',
                terms: 'תשלום 50% מראש, 50% בסיום הפרויקט',
                notes: 'הצעת מחיר מפורטת - ממתין לתגובת לקוח'
            },
            {
                id: 'quote_2',
                customer_id: 'customer_2',
                title: 'הצעת מחיר לתחזוקה שוטפת',
                amount: 500,
                status: 'approved',
                valid_until: '2024-12-31',
                description: 'תחזוקה שוטפת לאתר חנות אונליין',
                terms: 'תשלום חודשי',
                notes: 'אושר על ידי הלקוח - חוזה שנתי'
            }
        ];

        quotes.forEach(quote => {
            db.prepare(`
                INSERT INTO quotes (
                    id, customer_id, title, amount, status, valid_until,
                    description, terms, notes, created_date, updated_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            `).run(
                quote.id,
                quote.customer_id,
                quote.title,
                quote.amount,
                quote.status,
                quote.valid_until,
                quote.description,
                quote.terms,
                quote.notes,
                new Date().toISOString(),
                new Date().toISOString()
            );
        });

        // Insert sample payments
        console.log('💳 Creating sample payments...');
        const payments = [
            {
                id: 'payment_1',
                customer_id: 'customer_1',
                project_id: 'project_1',
                amount: 25000,
                payment_type: 'project_payment',
                status: 'completed',
                payment_method: 'bank_transfer',
                payment_date: '2024-09-01',
                invoice_number: 'INV-2024-001',
                description: 'תשלום ראשון - פרויקט CRM',
                notes: 'תשלום הושלם בהצלחה'
            },
            {
                id: 'payment_2',
                customer_id: 'customer_2',
                amount: 399,
                payment_type: 'domain_renewal',
                status: 'pending',
                payment_method: 'credit_card',
                due_date: '2024-11-15',
                description: 'חידוש דומיין shop-online.co.il',
                notes: 'תשלום חודשי'
            }
        ];

        payments.forEach(payment => {
            db.prepare(`
                INSERT INTO payments (
                    id, customer_id, project_id, amount, payment_type, status,
                    payment_method, payment_date, due_date, invoice_number,
                    description, notes, created_date, updated_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            `).run(
                payment.id,
                payment.customer_id,
                payment.project_id || null,
                payment.amount,
                payment.payment_type,
                payment.status,
                payment.payment_method,
                payment.payment_date || null,
                payment.due_date || null,
                payment.invoice_number || null,
                payment.description,
                payment.notes,
                new Date().toISOString(),
                new Date().toISOString()
            );
        });

        // Insert sample activities
        console.log('📝 Creating sample activities...');
        const activities = [
            {
                id: 'activity_1',
                customer_id: 'customer_1',
                activity_type: 'meeting',
                title: 'פגישה שבועית - עדכון פרויקט CRM',
                description: 'פגישה שבועית עם הלקוח לעדכון התקדמות הפרויקט',
                status: 'completed',
                priority: 'high',
                due_date: '2024-09-25',
                completed_date: '2024-09-25',
                assigned_to: 'דוד כהן',
                duration_minutes: 60,
                outcome: 'הלקוח מרוצה מההתקדמות',
                notes: 'פגישה מוצלחת - המשך לפי התכנון'
            },
            {
                id: 'activity_2',
                customer_id: 'customer_3',
                activity_type: 'call',
                title: 'שיחה עם לקוח חדש - משרד עורכי דין',
                description: 'שיחה ראשונית עם לקוח חדש לבדיקת צרכים',
                status: 'pending',
                priority: 'medium',
                due_date: '2024-09-30',
                assigned_to: 'מיכל רוזן',
                duration_minutes: 30,
                notes: 'להתקשר בשעה 14:00'
            }
        ];

        activities.forEach(activity => {
            db.prepare(`
                INSERT INTO activities (
                    id, customer_id, activity_type, title, description, status,
                    priority, due_date, completed_date, assigned_to, duration_minutes,
                    outcome, notes, created_date, updated_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            `).run(
                activity.id,
                activity.customer_id,
                activity.activity_type,
                activity.title,
                activity.description,
                activity.status,
                activity.priority,
                activity.due_date,
                activity.completed_date || null,
                activity.assigned_to,
                activity.duration_minutes,
                activity.outcome || null,
                activity.notes,
                new Date().toISOString(),
                new Date().toISOString()
            );
        });

        // Insert sample reminders
        console.log('⏰ Creating sample reminders...');
        const reminders = [
            {
                id: 'reminder_1',
                customer_id: 'customer_1',
                domain_id: 'domain_1',
                reminder_type: 'domain_renewal',
                title: 'חידוש דומיין tech-advanced.co.il',
                description: 'דומיין יפוג בעוד 30 ימים',
                reminder_date: '2024-11-15',
                status: 'pending',
                priority: 'high',
                recurring: 'yearly',
                notes: 'חידוש אוטומטי מופעל'
            },
            {
                id: 'reminder_2',
                customer_id: 'customer_2',
                reminder_type: 'payment_due',
                title: 'תשלום חודשי חנות אונליין',
                description: 'תשלום חודשי לתחזוקה שוטפת',
                reminder_date: '2024-10-01',
                status: 'completed',
                priority: 'medium',
                recurring: 'monthly',
                completed_date: '2024-10-01',
                notes: 'תשלום הושלם'
            }
        ];

        reminders.forEach(reminder => {
            db.prepare(`
                INSERT INTO reminders (
                    id, customer_id, domain_id, reminder_type, title, description,
                    reminder_date, status, priority, recurring, completed_date,
                    notes, created_date, updated_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            `).run(
                reminder.id,
                reminder.customer_id,
                reminder.domain_id || null,
                reminder.reminder_type,
                reminder.title,
                reminder.description,
                reminder.reminder_date,
                reminder.status,
                reminder.priority,
                reminder.recurring,
                reminder.completed_date || null,
                reminder.notes,
                new Date().toISOString(),
                new Date().toISOString()
            );
        });

        // Insert sample email services
        console.log('📧 Creating sample email services...');
        const emailServices = [
            {
                id: 'email_service_1',
                domain_id: 'domain_1',
                service_type: 'email_accounts',
                service_name: 'חשבונות אימייל עסקיים',
                provider: 'Google Workspace',
                status: 'active',
                start_date: '2024-01-01',
                end_date: '2024-12-31',
                monthly_cost: 50,
                features: '10 חשבונות אימייל, 30GB אחסון לכל חשבון',
                notes: 'שירות פעיל - חידוש אוטומטי'
            },
            {
                id: 'email_service_2',
                domain_id: 'domain_3',
                service_type: 'hosting',
                service_name: 'אירוח אתר חנות אונליין',
                provider: 'CloudFlare',
                status: 'active',
                start_date: '2024-08-01',
                end_date: '2025-07-31',
                monthly_cost: 100,
                features: 'SSL, CDN, אבטחה מתקדמת',
                notes: 'שירות אירוח מתקדם עם CDN'
            }
        ];

        emailServices.forEach(service => {
            db.prepare(`
                INSERT INTO email_services (
                    id, domain_id, service_type, service_name, provider, status,
                    start_date, end_date, monthly_cost, features, notes,
                    created_date, updated_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            `).run(
                service.id,
                service.domain_id,
                service.service_type,
                service.service_name,
                service.provider,
                service.status,
                service.start_date,
                service.end_date,
                service.monthly_cost,
                service.features,
                service.notes,
                new Date().toISOString(),
                new Date().toISOString()
            );
        });

        // Insert sample technical services
        console.log('⚙️ Creating sample technical services...');
        const technicalServices = [
            {
                id: 'tech_service_1',
                domain_id: 'domain_1',
                service_type: 'ssl_certificate',
                service_name: 'תעודת SSL מתקדמת',
                provider: 'DigiCert',
                status: 'active',
                start_date: '2024-01-01',
                end_date: '2024-12-31',
                monthly_cost: 25,
                configuration: 'Wildcard SSL, 256-bit encryption',
                notes: 'תעודת SSL לכל הדומיינים'
            },
            {
                id: 'tech_service_2',
                domain_id: 'domain_3',
                service_type: 'monitoring',
                service_name: 'מערכת ניטור אתר',
                provider: 'UptimeRobot',
                status: 'active',
                start_date: '2024-08-01',
                monthly_cost: 15,
                configuration: 'ניטור כל 5 דקות, התראות SMS ואימייל',
                notes: 'ניטור 24/7 של זמינות האתר'
            }
        ];

        technicalServices.forEach(service => {
            db.prepare(`
                INSERT INTO technical_services (
                    id, domain_id, service_type, service_name, provider, status,
                    start_date, end_date, monthly_cost, configuration, notes,
                    created_date, updated_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            `).run(
                service.id,
                service.domain_id,
                service.service_type,
                service.service_name,
                service.provider,
                service.status,
                service.start_date,
                service.end_date || null,
                service.monthly_cost,
                service.configuration,
                service.notes,
                new Date().toISOString(),
                new Date().toISOString()
            );
        });

        console.log('✅ Comprehensive data seeding completed successfully!');
        console.log(`📊 Created:`);
        console.log(`   - ${customers.length} customers`);
        console.log(`   - ${domains.length} domains`);
        console.log(`   - ${projects.length} projects`);
        console.log(`   - ${quotes.length} quotes`);
        console.log(`   - ${payments.length} payments`);
        console.log(`   - ${activities.length} activities`);
        console.log(`   - ${reminders.length} reminders`);
        console.log(`   - ${emailServices.length} email services`);
        console.log(`   - ${technicalServices.length} technical services`);

    } catch (error) {
        console.error('❌ Error seeding comprehensive data:', error);
    }
}

// Run the seeding if this file is executed directly
if (require.main === module) {
    seedComprehensiveData();
}

module.exports = { seedComprehensiveData };
