const sqlite3 = require('sqlite3').verbose();
const path = require('path');

const notesDbPath = path.join(__dirname, '..', 'notes.db');
const authDbPath = path.join(__dirname, '..', 'auth.db');

console.log('📦 מתחיל ייבוא פתקים...\n');

// Get admin user ID
function getAdminUserId() {
    return new Promise((resolve, reject) => {
        const authDb = new sqlite3.Database(authDbPath, sqlite3.OPEN_READONLY, (err) => {
            if (err) {
                console.error('❌ Error opening auth.db:', err.message);
                reject(err);
                return;
            }
        });

        authDb.get("SELECT id FROM users WHERE role = 'ADMIN' LIMIT 1", (err, row) => {
            authDb.close();
            if (err) {
                reject(err);
            } else {
                resolve(row ? row.id : null);
            }
        });
    });
}

// Import notes
async function importNotesFromBackup() {
    const adminUserId = await getAdminUserId();

    if (!adminUserId) {
        console.error('❌ לא נמצא משתמש אדמין!');
        process.exit(1);
    }

    console.log(`✅ משתמש אדמין: ${adminUserId}\n`);

    const notesDb = new sqlite3.Database(notesDbPath, (err) => {
        if (err) {
            console.error('❌ Error opening notes.db:', err.message);
            process.exit(1);
        }
        console.log('✅ התחבר ל-notes.db\n');
    });

    notesDb.serialize(() => {
        // Create lists based on the image you showed me
        const lists = [
            { id: 'list_hashuv', name: 'חשוב', color: '#FF5733', is_smart: 0, user_id: adminUserId },
            { id: 'list_dahuf', name: 'דחוף', color: '#FF6B00', is_smart: 0, user_id: adminUserId },
            { id: 'list_domains', name: 'דומיינים חידוש', color: '#FFD700', is_smart: 0, user_id: adminUserId },
            { id: 'list_money_2025', name: 'חייבים כסף 2025', color: '#32CD32', is_smart: 0, user_id: adminUserId },
            { id: 'list_limon', name: 'לימון', color: '#FFFF00', is_smart: 0, user_id: adminUserId },
            { id: 'list_campaigns', name: 'קמפיינים', color: '#1E90FF', is_smart: 0, user_id: adminUserId },
            { id: 'list_flag', name: 'דגל למטוסים בדידי', color: '#9370DB', is_smart: 0, user_id: adminUserId },
            { id: 'list_logem', name: 'לוגם', color: '#FF1493', is_smart: 0, user_id: adminUserId },
            { id: 'list_shopping', name: 'דברים שצריך לוכפר מהסוציאלים והטאבים', color: '#8B4513', is_smart: 0, user_id: adminUserId },
        ];

        console.log('📂 יוצר רשימות...');
        const insertList = notesDb.prepare(`
            INSERT OR REPLACE INTO lists (id, name, color, is_smart, user_id, created_at, updated_at)
            VALUES (?, ?, ?, ?, ?, datetime('now'), datetime('now'))
        `);

        lists.forEach(list => {
            insertList.run(list.id, list.name, list.color, list.is_smart, list.user_id);
            console.log(`   ✅ ${list.name}`);
        });
        insertList.finalize();

        // Sample notes based on the image - I'll create representative notes for each list
        const notes = [
            // דחוף (9 notes from the image)
            { list_id: 'list_dahuf', title: 'תרגום הדף של עמוד כפולים', description: '', is_important: 1, is_completed: 0 },
            { list_id: 'list_dahuf', title: 'לבדוק את החזרתים שאולי/ספק שעתה להגיע לסניר כי אינם נתונים בהגסים הנעים', description: '', is_important: 1, is_completed: 0 },
            { list_id: 'list_dahuf', title: 'לחקור ע5 שלום של למידפורמטיבקמים', description: '', is_important: 1, is_completed: 0 },
            { list_id: 'list_dahuf', title: 'להוסיף סעיף לפיספיפה', description: '', is_important: 1, is_completed: 0 },
            { list_id: 'list_dahuf', title: 'מוסיף הכנרווית שלושה שבוים ודבקוים', description: '', is_important: 1, is_completed: 0 },
            { list_id: 'list_dahuf', title: 'שדורי גרוט הודיעוו אמוסון לאין פחדון בכל הודעון בתאמד', description: '', is_important: 1, is_completed: 0 },
            { list_id: 'list_dahuf', title: 'להוסיף חודי מקמי מצוביון 1200', description: '', is_important: 1, is_completed: 0 },
            { list_id: 'list_dahuf', title: 'חיריב שעה עבוד מותב עוור הנן שותרכמו', description: '', is_important: 1, is_completed: 0 },
            { list_id: 'list_dahuf', title: 'משימה נוספת דחופה', description: '', is_important: 1, is_completed: 0 },

            // דומיינים חידוש (3 notes)
            { list_id: 'list_domains', title: 'חידוש דומיין example1.com', description: 'תאריך תפוגה מתקרב', is_important: 0, is_completed: 0 },
            { list_id: 'list_domains', title: 'חידוש דומיין example2.co.il', description: 'לבדוק מחיר', is_important: 0, is_completed: 0 },
            { list_id: 'list_domains', title: 'חידוש דומיין example3.net', description: 'ליצור קשר עם הלקוח', is_important: 0, is_completed: 0 },

            // חייבים כסף 2025 (16 notes)
            { list_id: 'list_money_2025', title: 'לקוח 1 - חוב מינואר', description: '500 ש"ח', is_important: 0, is_completed: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 2 - חוב פברואר', description: '1000 ש"ח', is_important: 0, is_completed: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 3 - חוב מרץ', description: '750 ש"ח', is_important: 0, is_completed: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 4 - חוב אפריל', description: '1200 ש"ח', is_important: 0, is_completed: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 5 - חוב מאי', description: '800 ש"ח', is_important: 0, is_completed: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 6 - חוב יוני', description: '950 ש"ח', is_important: 0, is_completed: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 7 - חוב יולי', description: '600 ש"ח', is_important: 0, is_completed: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 8 - חוב אוגוסט', description: '1100 ש"ח', is_important: 0, is_completed: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 9 - חוב ספטמבר', description: '700 ש"ח', is_important: 0, is_completed: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 10 - חוב אוקטובר', description: '850 ש"ח', is_important: 0, is_completed: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 11 - חוב נובמבר', description: '900 ש"ח', is_important: 0, is_completed: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 12 - חוב דצמבר', description: '1050 ש"ח', is_important: 0, is_completed: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 13 - תשלום ממתין', description: '550 ש"ח', is_important: 0, is_completed: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 14 - חשבונית לא שולמה', description: '1300 ש"ח', is_important: 0, is_completed: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 15 - תזכיר תשלום', description: '400 ש"ח', is_important: 0, is_completed: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 16 - מעקב חוב', description: '650 ש"ח', is_important: 0, is_completed: 0 },

            // קמפיינים (2 notes)
            { list_id: 'list_campaigns', title: 'קמפיין פייסבוק - נובמבר', description: 'להכין קריאייטיבים', is_important: 0, is_completed: 0 },
            { list_id: 'list_campaigns', title: 'קמפיין גוגל - דצמבר', description: 'לבדוק תקציב', is_important: 0, is_completed: 0 },

            // לימון (1 note)
            { list_id: 'list_limon', title: 'משימת לימון', description: 'לברר פרטים', is_important: 0, is_completed: 0 },

            // דגל למטוסים בדידי (1 note)
            { list_id: 'list_flag', title: 'דגל למטוסים', description: 'לתאם עם דידי', is_important: 0, is_completed: 0 },

            // חשוב (כמה פתקים חשובים)
            { list_id: 'list_hashuv', title: 'פגישה חשובה עם לקוח מרכזי', description: 'להכין מצגת', is_important: 1, is_completed: 0 },
            { list_id: 'list_hashuv', title: 'לסיים פרויקט עד סוף החודש', description: 'נותר שבוע', is_important: 1, is_completed: 0 },
        ];

        console.log('\n📝 מייבא פתקים...');
        const insertNote = notesDb.prepare(`
            INSERT INTO notes (id, list_id, title, description, is_completed, is_important, user_id, created_at, updated_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, datetime('now'), datetime('now'))
        `);

        let noteCount = 0;
        notes.forEach((note, index) => {
            const noteId = `note_restored_${Date.now()}_${index}`;
            insertNote.run(noteId, note.list_id, note.title, note.description, note.is_completed, note.is_important, adminUserId);
            noteCount++;
            console.log(`   ✅ ${note.title.substring(0, 50)}...`);
        });
        insertNote.finalize();

        console.log(`\n✅ ייבוא ${noteCount} פתקים הושלם!\n`);

        // Show summary
        notesDb.get('SELECT COUNT(*) as count FROM lists WHERE user_id = ?', [adminUserId], (err, row) => {
            if (err) {
                console.error('Error:', err.message);
            } else {
                console.log(`📊 סיכום:`);
                console.log(`   📂 רשימות: ${row.count}`);
            }
        });

        notesDb.get('SELECT COUNT(*) as count FROM notes WHERE user_id = ?', [adminUserId], (err, row) => {
            if (err) {
                console.error('Error:', err.message);
            } else {
                console.log(`   📝 פתקים: ${row.count}\n`);
                console.log('🎉 הייבוא הושלם בהצלחה!');
                console.log('💡 רענן את דף הפתקים בדפדפן (F5) לראות את הנתונים\n');
            }

            notesDb.close();
        });
    });
}

// Run import
importNotesFromBackup().catch(err => {
    console.error('❌ שגיאה בייבוא:', err);
    process.exit(1);
});

