import { phones } from '../src/lib/data';
import { getDatabase, initDatabase } from '../src/lib/database';

const migrateData = async () => {
    try {
        console.log('Starting migration to SQLite...');

        // Initialize database
        await initDatabase();
        const db = await getDatabase();

        // Clear existing data (if table exists)
        await new Promise<void>((resolve, reject) => {
            db.run('DELETE FROM phones', (err) => {
                if (err) {
                    console.log('No existing data to clear (table may not exist yet)');
                } else {
                    console.log('Cleared existing data');
                }
                resolve();
            });
        });

        // Insert sample data
        for (const phone of phones) {
            const id = Date.now().toString() + Math.random().toString(36).substr(2, 9);
            const now = new Date().toISOString();

            await new Promise<void>((resolve, reject) => {
                db.run(`INSERT INTO phones (id, phone, domain, website, owner, email, package, amount, status, priority, startDate, nextPayment, notes, createdAt, updatedAt) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`, [
                    id, phone.phone, phone.domain, phone.website, phone.owner, phone.email,
                    phone.package, phone.amount, phone.status, phone.priority,
                    phone.startDate || new Date().toISOString().split('T')[0], phone.nextPayment, phone.notes || '', now, now
                ], (err) => {
                    if (err) {
                        console.error('Error inserting phone:', err);
                        reject(err);
                    } else {
                        console.log(`Inserted phone: ${phone.phone}`);
                        resolve();
                    }
                });
            });
        }

        console.log('✅ Data migrated successfully!');
        console.log(`📊 Migrated ${phones.length} phones to SQLite`);

    } catch (error) {
        console.error('❌ Migration failed:', error);
    }
};

// Run migration if this file is executed directly
if (require.main === module) {
    migrateData();
}

export default migrateData;
