const Database = require('better-sqlite3');
const path = require('path');

console.log('🔧 Creating tasks table in comprehensive.db...');

const dbPath = path.join(process.cwd(), 'comprehensive.db');
const db = new Database(dbPath);

try {
    // Create tasks table
    db.exec(`
        CREATE TABLE IF NOT EXISTS tasks (
            id TEXT PRIMARY KEY,
            title TEXT NOT NULL,
            description TEXT,
            status TEXT DEFAULT 'pending',
            priority TEXT DEFAULT 'medium',
            category_id TEXT,
            customer_id TEXT,
            project_id TEXT,
            assigned_to TEXT,
            due_date TEXT,
            completed_at TEXT,
            created_at TEXT DEFAULT CURRENT_TIMESTAMP,
            updated_at TEXT DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (customer_id) REFERENCES customers(id),
            FOREIGN KEY (project_id) REFERENCES projects(id)
        );
    `);
    console.log('✅ Tasks table created');

    // Create tasks_categories table
    db.exec(`
        CREATE TABLE IF NOT EXISTS tasks_categories (
            id TEXT PRIMARY KEY,
            name TEXT NOT NULL,
            color TEXT DEFAULT '#3B82F6',
            icon TEXT DEFAULT '📋',
            created_at TEXT DEFAULT CURRENT_TIMESTAMP
        );
    `);
    console.log('✅ Tasks categories table created');

    // Insert default categories
    const categories = [
        { id: 'cat_1', name: 'כללי', color: '#3B82F6', icon: '📋' },
        { id: 'cat_2', name: 'דחוף', color: '#EF4444', icon: '🔥' },
        { id: 'cat_3', name: 'פגישה', color: '#10B981', icon: '🤝' },
        { id: 'cat_4', name: 'תמיכה', color: '#F59E0B', icon: '🛠️' },
        { id: 'cat_5', name: 'פיתוח', color: '#8B5CF6', icon: '💻' }
    ];

    const insertCategory = db.prepare(`
        INSERT OR IGNORE INTO tasks_categories (id, name, color, icon, created_at)
        VALUES (?, ?, ?, ?, datetime('now'))
    `);

    categories.forEach(cat => {
        insertCategory.run(cat.id, cat.name, cat.color, cat.icon);
    });
    console.log('✅ Default categories inserted');

    // Create tasks_full_view
    db.exec(`
        CREATE VIEW IF NOT EXISTS tasks_full_view AS
        SELECT 
            t.*,
            c.name as customer_name,
            p.title as project_name,
            cat.name as category_name,
            cat.color as category_color,
            cat.icon as category_icon
        FROM tasks t
        LEFT JOIN customers c ON t.customer_id = c.id
        LEFT JOIN projects p ON t.project_id = p.id
        LEFT JOIN tasks_categories cat ON t.category_id = cat.id;
    `);
    console.log('✅ Tasks full view created');

    // Create indexes for performance
    db.exec(`
        CREATE INDEX IF NOT EXISTS idx_tasks_status ON tasks(status);
        CREATE INDEX IF NOT EXISTS idx_tasks_priority ON tasks(priority);
        CREATE INDEX IF NOT EXISTS idx_tasks_customer ON tasks(customer_id);
        CREATE INDEX IF NOT EXISTS idx_tasks_project ON tasks(project_id);
        CREATE INDEX IF NOT EXISTS idx_tasks_due_date ON tasks(due_date);
    `);
    console.log('✅ Indexes created');

    console.log('');
    console.log('🎉 Tasks module initialized successfully!');

    db.close();
} catch (error) {
    console.error('❌ Error initializing tasks:', error);
    db.close();
    process.exit(1);
}

