const Database = require('better-sqlite3');
const path = require('path');

console.log('🔧 Creating events table in comprehensive.db...');

const dbPath = path.join(process.cwd(), 'comprehensive.db');
const db = new Database(dbPath);

try {
    // Create events table
    db.exec(`
        CREATE TABLE IF NOT EXISTS events (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            title TEXT NOT NULL,
            description TEXT,
            start_datetime TEXT NOT NULL,
            end_datetime TEXT,
            location TEXT,
            customer_id TEXT,
            domain_id TEXT,
            project_id TEXT,
            quote_id TEXT,
            task_id TEXT,
            event_type TEXT DEFAULT 'meeting',
            status TEXT DEFAULT 'scheduled',
            attendees TEXT,
            reminder_minutes INTEGER DEFAULT 15,
            created_by TEXT,
            created_at TEXT DEFAULT CURRENT_TIMESTAMP,
            updated_at TEXT DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (customer_id) REFERENCES customers(id),
            FOREIGN KEY (domain_id) REFERENCES domains(id),
            FOREIGN KEY (project_id) REFERENCES projects(id),
            FOREIGN KEY (task_id) REFERENCES tasks(id)
        );
    `);
    console.log('✅ Events table created');

    // Create indexes for performance
    db.exec(`
        CREATE INDEX IF NOT EXISTS idx_events_customer ON events(customer_id);
        CREATE INDEX IF NOT EXISTS idx_events_project ON events(project_id);
        CREATE INDEX IF NOT EXISTS idx_events_start ON events(start_datetime);
        CREATE INDEX IF NOT EXISTS idx_events_type ON events(event_type);
        CREATE INDEX IF NOT EXISTS idx_events_status ON events(status);
    `);
    console.log('✅ Indexes created');

    // Count events
    const count = db.prepare('SELECT COUNT(*) as count FROM events').get();
    console.log(`📊 Total events: ${count.count}`);

    console.log('');
    console.log('🎉 Events table initialized successfully!');

    db.close();
} catch (error) {
    console.error('❌ Error initializing events:', error);
    db.close();
    process.exit(1);
}

