import sqlite3 from 'sqlite3';
import path from 'path';

const initComprehensiveDB = async (): Promise<void> => {
    return new Promise((resolve, reject) => {
        const dbPath = path.join(process.cwd(), 'comprehensive.db');
        const db = new sqlite3.Database(dbPath, (err) => {
            if (err) {
                console.error('Error opening comprehensive.db:', err);
                reject(err);
                return;
            }
            console.log('Connected to comprehensive.db at:', dbPath);
        });

        // Create customers table
        const customersTable = `
            CREATE TABLE IF NOT EXISTS customers (
                id TEXT PRIMARY KEY,
                name TEXT NOT NULL,
                email TEXT,
                phone TEXT,
                company TEXT,
                status TEXT DEFAULT 'lead',
                stage TEXT DEFAULT 'quote',
                assigned_sales_rep TEXT,
                value REAL DEFAULT 0,
                notes TEXT,
                created_date TEXT DEFAULT CURRENT_TIMESTAMP,
                last_activity TEXT DEFAULT CURRENT_TIMESTAMP,
                updated_at TEXT DEFAULT CURRENT_TIMESTAMP
            )
        `;

        // Create domains table
        const domainsTable = `
            CREATE TABLE IF NOT EXISTS domains (
                id TEXT PRIMARY KEY,
                name TEXT NOT NULL,
                customer_id TEXT,
                status TEXT DEFAULT 'active',
                registration_date TEXT,
                expiry_date TEXT,
                provider TEXT,
                price REAL,
                notes TEXT,
                created_at TEXT DEFAULT CURRENT_TIMESTAMP,
                updated_at TEXT DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (customer_id) REFERENCES customers(id)
            )
        `;

        // Create projects table
        const projectsTable = `
            CREATE TABLE IF NOT EXISTS projects (
                id TEXT PRIMARY KEY,
                name TEXT NOT NULL,
                customer_id TEXT,
                status TEXT DEFAULT 'active',
                start_date TEXT,
                end_date TEXT,
                budget REAL,
                description TEXT,
                created_at TEXT DEFAULT CURRENT_TIMESTAMP,
                updated_at TEXT DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (customer_id) REFERENCES customers(id)
            )
        `;

        // Create quotes table
        const quotesTable = `
            CREATE TABLE IF NOT EXISTS quotes (
                id TEXT PRIMARY KEY,
                customer_id TEXT,
                project_id TEXT,
                amount REAL,
                status TEXT DEFAULT 'draft',
                created_at TEXT DEFAULT CURRENT_TIMESTAMP,
                updated_at TEXT DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (customer_id) REFERENCES customers(id),
                FOREIGN KEY (project_id) REFERENCES projects(id)
            )
        `;

        db.serialize(() => {
            db.run(customersTable, (err) => {
                if (err) {
                    console.error('Error creating customers table:', err);
                } else {
                    console.log('Customers table created successfully or already exists');
                }
            });

            db.run(domainsTable, (err) => {
                if (err) {
                    console.error('Error creating domains table:', err);
                } else {
                    console.log('Domains table created successfully or already exists');
                }
            });

            db.run(projectsTable, (err) => {
                if (err) {
                    console.error('Error creating projects table:', err);
                } else {
                    console.log('Projects table created successfully or already exists');
                }
            });

            db.run(quotesTable, (err) => {
                if (err) {
                    console.error('Error creating quotes table:', err);
                } else {
                    console.log('Quotes table created successfully or already exists');
                }
            });
        });

        db.close((err) => {
            if (err) {
                console.error('Error closing comprehensive.db:', err);
                reject(err);
            } else {
                console.log('comprehensive.db initialized successfully');
                resolve();
            }
        });
    });
};

const initDomainsDB = async (): Promise<void> => {
    return new Promise((resolve, reject) => {
        const dbPath = path.join(process.cwd(), 'domains.db');
        const db = new sqlite3.Database(dbPath, (err) => {
            if (err) {
                console.error('Error opening domains.db:', err);
                reject(err);
                return;
            }
            console.log('Connected to domains.db at:', dbPath);
        });

        // Create customers table (duplicate for domains.db)
        const customersTable = `
            CREATE TABLE IF NOT EXISTS customers (
                id TEXT PRIMARY KEY,
                name TEXT NOT NULL,
                email TEXT,
                phone TEXT,
                company TEXT,
                status TEXT DEFAULT 'lead',
                stage TEXT DEFAULT 'quote',
                assigned_sales_rep TEXT,
                value REAL DEFAULT 0,
                notes TEXT,
                created_date TEXT DEFAULT CURRENT_TIMESTAMP,
                last_activity TEXT DEFAULT CURRENT_TIMESTAMP,
                updated_at TEXT DEFAULT CURRENT_TIMESTAMP
            )
        `;

        // Create domains table
        const domainsTable = `
            CREATE TABLE IF NOT EXISTS domains (
                id TEXT PRIMARY KEY,
                name TEXT NOT NULL,
                customer_id TEXT,
                status TEXT DEFAULT 'active',
                registration_date TEXT,
                expiry_date TEXT,
                provider TEXT,
                price REAL,
                notes TEXT,
                created_at TEXT DEFAULT CURRENT_TIMESTAMP,
                updated_at TEXT DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (customer_id) REFERENCES customers(id)
            )
        `;

        db.serialize(() => {
            db.run(customersTable, (err) => {
                if (err) {
                    console.error('Error creating customers table in domains.db:', err);
                } else {
                    console.log('Customers table created successfully in domains.db');
                }
            });

            db.run(domainsTable, (err) => {
                if (err) {
                    console.error('Error creating domains table in domains.db:', err);
                } else {
                    console.log('Domains table created successfully in domains.db');
                }
            });
        });

        db.close((err) => {
            if (err) {
                console.error('Error closing domains.db:', err);
                reject(err);
            } else {
                console.log('domains.db initialized successfully');
                resolve();
            }
        });
    });
};

const main = async () => {
    try {
        console.log('Initializing all databases...');

        await initComprehensiveDB();
        await initDomainsDB();

        console.log('✅ All databases initialized successfully!');
        process.exit(0);
    } catch (error) {
        console.error('❌ Error initializing databases:', error);
        process.exit(1);
    }
};

if (require.main === module) {
    main();
}

export { initComprehensiveDB, initDomainsDB };
