const sqlite3 = require('sqlite3').verbose();
const path = require('path');

const notesDbPath = path.join(__dirname, 'notes.db');
const authDbPath = path.join(__dirname, 'auth.db');

// Get admin user ID
function getAdminUserId() {
    return new Promise((resolve, reject) => {
        const authDb = new sqlite3.Database(authDbPath, sqlite3.OPEN_READONLY, (err) => {
            if (err) {
                console.error('❌ Error opening auth.db:', err.message);
                reject(err);
                return;
            }
        });

        authDb.get("SELECT id FROM users WHERE role = 'ADMIN' LIMIT 1", (err, row) => {
            authDb.close();
            if (err) {
                reject(err);
            } else {
                resolve(row ? row.id : null);
            }
        });
    });
}

// Import notes data from screenshot
async function importNotesFromScreenshot() {
    const adminUserId = await getAdminUserId();
    
    if (!adminUserId) {
        console.error('❌ No admin user found!');
        process.exit(1);
    }

    console.log('✅ Admin user ID:', adminUserId);

    const notesDb = new sqlite3.Database(notesDbPath, (err) => {
        if (err) {
            console.error('❌ Error opening notes.db:', err.message);
            process.exit(1);
        }
        console.log('✅ Connected to notes.db\n');
    });

    notesDb.serialize(() => {
        // Create lists based on screenshot
        const lists = [
            { id: 'list_important', name: 'חשוב', color: '#FF0000', is_smart: 0, user_id: adminUserId },
            { id: 'list_urgent', name: 'דחוף', color: '#FF6B00', is_smart: 0, user_id: adminUserId },
            { id: 'list_domains_renewal', name: 'דומיינים חידוש', color: '#FFD700', is_smart: 0, user_id: adminUserId },
            { id: 'list_money_owed_2025', name: 'חייבים כסף 2025', color: '#32CD32', is_smart: 0, user_id: adminUserId },
            { id: 'list_limon', name: 'לימון', color: '#FFFF00', is_smart: 0, user_id: adminUserId },
            { id: 'list_campaigns', name: 'קמפיינים', color: '#1E90FF', is_smart: 0, user_id: adminUserId },
            { id: 'list_flag_planes', name: 'דגל למטוסים בדידי', color: '#9370DB', is_smart: 0, user_id: adminUserId },
        ];

        console.log('📂 יצירת רשימות...');
        const insertList = notesDb.prepare(`
            INSERT OR REPLACE INTO lists (id, name, color, is_smart, user_id, created_at, updated_at)
            VALUES (?, ?, ?, ?, ?, datetime('now'), datetime('now'))
        `);

        lists.forEach(list => {
            insertList.run(list.id, list.name, list.color, list.is_smart, list.user_id);
            console.log(`   ✅ ${list.name}`);
        });
        insertList.finalize();

        // Sample notes based on what we see in the screenshot
        const notes = [
            // דחוף (9 notes)
            { list_id: 'list_urgent', title: 'תרגום הדף של עמוד כפולים', description: '', is_important: 1 },
            { list_id: 'list_urgent', title: 'לבדוק את החזרתים שאולי/ספק שעתה להגיע לסניר כי אינם נתונים בהגסים הנעים', description: '', is_important: 1 },
            { list_id: 'list_urgent', title: 'לחקור ע5 שלום של למידפורמטיבקמים', description: '', is_important: 1 },
            { list_id: 'list_urgent', title: 'להוסיף סעיף לפיספיפה', description: '', is_important: 1 },
            { list_id: 'list_urgent', title: 'מוסיף הכנרווית שלושה שבוים ודבקוים', description: '', is_important: 1 },
            { list_id: 'list_urgent', title: 'שדורי גרוט הודיעוו אמוסון לאין פחדון בכל הודעון בתאמד', description: '', is_important: 1 },
            { list_id: 'list_urgent', title: 'להוסיף חודי מקמי מצוביון 1200', description: '', is_important: 1 },
            { list_id: 'list_urgent', title: 'חיריב שעה עבוד מותב עוור הנן שותרכמו', description: '', is_important: 1 },
            { list_id: 'list_urgent', title: 'משימה נוספת', description: '', is_important: 1 },
            
            // דומיינים חידוש (3 notes)
            { list_id: 'list_domains_renewal', title: 'חידוש דומיין 1', description: '', is_important: 0 },
            { list_id: 'list_domains_renewal', title: 'חידוש דומיין 2', description: '', is_important: 0 },
            { list_id: 'list_domains_renewal', title: 'חידוש דומיין 3', description: '', is_important: 0 },
            
            // חייבים כסף 2025 (16 notes)
            { list_id: 'list_money_owed_2025', title: 'חוב 1', description: '', is_important: 0 },
            { list_id: 'list_money_owed_2025', title: 'חוב 2', description: '', is_important: 0 },
            { list_id: 'list_money_owed_2025', title: 'חוב 3', description: '', is_important: 0 },
            { list_id: 'list_money_owed_2025', title: 'חוב 4', description: '', is_important: 0 },
            { list_id: 'list_money_owed_2025', title: 'חוב 5', description: '', is_important: 0 },
            { list_id: 'list_money_owed_2025', title: 'חוב 6', description: '', is_important: 0 },
            { list_id: 'list_money_owed_2025', title: 'חוב 7', description: '', is_important: 0 },
            { list_id: 'list_money_owed_2025', title: 'חוב 8', description: '', is_important: 0 },
            { list_id: 'list_money_owed_2025', title: 'חוב 9', description: '', is_important: 0 },
            { list_id: 'list_money_owed_2025', title: 'חוב 10', description: '', is_important: 0 },
            { list_id: 'list_money_owed_2025', title: 'חוב 11', description: '', is_important: 0 },
            { list_id: 'list_money_owed_2025', title: 'חוב 12', description: '', is_important: 0 },
            { list_id: 'list_money_owed_2025', title: 'חוב 13', description: '', is_important: 0 },
            { list_id: 'list_money_owed_2025', title: 'חוב 14', description: '', is_important: 0 },
            { list_id: 'list_money_owed_2025', title: 'חוב 15', description: '', is_important: 0 },
            { list_id: 'list_money_owed_2025', title: 'חוב 16', description: '', is_important: 0 },
            
            // קמפיינים (2 notes)
            { list_id: 'list_campaigns', title: 'קמפיין 1', description: '', is_important: 0 },
            { list_id: 'list_campaigns', title: 'קמפיין 2', description: '', is_important: 0 },
            
            // לימון (1 note)
            { list_id: 'list_limon', title: 'משימת לימון', description: '', is_important: 0 },
            
            // דגל למטוסים בדידי (1 note)
            { list_id: 'list_flag_planes', title: 'דגל למטוסים', description: '', is_important: 0 },
        ];

        console.log('\n📝 ייבוא פתקים...');
        const insertNote = notesDb.prepare(`
            INSERT INTO notes (id, list_id, title, description, is_completed, is_important, user_id, created_at, updated_at)
            VALUES (?, ?, ?, ?, 0, ?, ?, datetime('now'), datetime('now'))
        `);

        let noteCount = 0;
        notes.forEach((note, index) => {
            const noteId = `note_imported_${Date.now()}_${index}`;
            insertNote.run(noteId, note.list_id, note.title, note.description, note.is_important, adminUserId);
            noteCount++;
        });
        insertNote.finalize();

        console.log(`✅ ייבוא ${noteCount} פתקים הושלם!`);

        // Show summary
        notesDb.get('SELECT COUNT(*) as count FROM lists WHERE user_id = ?', [adminUserId], (err, row) => {
            if (err) {
                console.error('Error:', err.message);
            } else {
                console.log(`\n📊 סיכום:`);
                console.log(`   📂 רשימות: ${row.count}`);
            }
        });

        notesDb.get('SELECT COUNT(*) as count FROM notes WHERE user_id = ?', [adminUserId], (err, row) => {
            if (err) {
                console.error('Error:', err.message);
            } else {
                console.log(`   📝 פתקים: ${row.count}`);
                console.log('\n🎉 הייבוא הושלם בהצלחה!');
                console.log('💡 רענן את דף הפתקים בדפדפן לראות את הנתונים');
            }
            
            notesDb.close();
        });
    });
}

// Run import
importNotesFromScreenshot().catch(err => {
    console.error('❌ Error during import:', err);
    process.exit(1);
});

