/**
 * Import Notes from PDF files
 * ייבוא פתקים מקבצי PDF
 */

import fs from 'fs';
import path from 'path';
import pdf from 'pdf-parse';
import sqlite3 from 'sqlite3';
import { open } from 'sqlite';

interface ParsedNote {
    title: string;
    description?: string;
    done: boolean;
    important: boolean;
    listName: string;
}

// Path to PDFs
const PDF_DIR = path.join(process.cwd(), '.vscode/project log + md/פטקים');
const DB_PATH = path.join(process.cwd(), 'notes.db');

// PDF to List mapping
const PDF_TO_LIST_MAP: Record<string, { name: string; icon: string; color: string }> = {
    'חשוב - ניהול משימות.pdf': {
        name: '⭐ חשוב',
        icon: '⭐',
        color: '#F39C12'
    },
    'חייבים כסף 2025 - ניהול משימות.pdf': {
        name: '💰 חייבים כסף 2025',
        icon: '💰',
        color: '#E74C3C'
    },
    'דחוף - ניהול משימות.pdf': {
        name: '🚨 דחוף',
        icon: '🚨',
        color: '#E74C3C'
    },
    'דומיינים חידוש - ניהול משימות.pdf': {
        name: '🌐 דומיינים חידוש',
        icon: '🌐',
        color: '#3498DB'
    }
};

function generateId(): string {
    return Date.now().toString() + Math.random().toString(36).substr(2, 9);
}

async function parsePdfToNotes(pdfPath: string): Promise<ParsedNote[]> {
    try {
        const dataBuffer = fs.readFileSync(pdfPath);
        const data = await pdf(dataBuffer);

        const text = data.text;
        const lines = text.split('\n').map(line => line.trim()).filter(line => line.length > 0);

        const notes: ParsedNote[] = [];
        const fileName = path.basename(pdfPath);
        const listInfo = PDF_TO_LIST_MAP[fileName];

        console.log(`\n📄 Processing: ${fileName}`);
        console.log(`📝 Total lines: ${lines.length}`);

        // Parse notes from text
        // Common patterns:
        // - Lines starting with ☐ or ☑ are tasks
        // - Lines with bullet points are items
        // - Checkmarks indicate completion

        for (const line of lines) {
            // Skip headers and irrelevant lines
            if (line.includes('ניהול משימות') || line.length < 3) continue;

            let title = line;
            let done = false;
            let important = false;

            // Check for checkboxes
            if (line.includes('☑') || line.includes('✓') || line.includes('✔')) {
                done = true;
                title = line.replace(/[☑✓✔]/g, '').trim();
            }

            if (line.includes('☐') || line.includes('□')) {
                done = false;
                title = line.replace(/[☐□]/g, '').trim();
            }

            // Check for importance markers
            if (line.includes('⭐') || line.includes('!') || line.includes('חשוב')) {
                important = true;
            }

            // Remove bullet points and clean up
            title = title.replace(/^[•\-\*]\s*/, '').trim();

            if (title.length > 0 && title.length < 200) {
                notes.push({
                    title,
                    done,
                    important,
                    listName: listInfo.name
                });
            }
        }

        console.log(`✅ Parsed ${notes.length} notes from ${fileName}`);
        return notes;

    } catch (error) {
        console.error(`❌ Error parsing ${pdfPath}:`, error);
        return [];
    }
}

async function importNotesToDb(notes: ParsedNote[]) {
    const db = await open({
        filename: DB_PATH,
        driver: sqlite3.Database
    });

    console.log('\n🔗 Connected to database');

    // Get or create lists
    const listMap = new Map<string, string>();

    for (const [pdfName, listInfo] of Object.entries(PDF_TO_LIST_MAP)) {
        // Check if list exists
        let list = await db.get('SELECT * FROM lists WHERE name = ?', [listInfo.name]);

        if (!list) {
            // Create list
            const id = generateId();
            await db.run(
                'INSERT INTO lists (id, name, icon, color, sort_order, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?)',
                [id, listInfo.name, listInfo.icon, listInfo.color, 0, new Date().toISOString(), new Date().toISOString()]
            );
            console.log(`📝 Created list: ${listInfo.name}`);
            listMap.set(listInfo.name, id);
        } else {
            listMap.set(listInfo.name, list.id);
        }
    }

    // Import notes
    let imported = 0;
    for (const note of notes) {
        const listId = listMap.get(note.listName);
        if (!listId) continue;

        const id = generateId();
        const now = new Date().toISOString();

        try {
            await db.run(
                `INSERT INTO notes (
                    id, list_id, title, description, done, important, flagged,
                    due_date, reminder_date, reminder_time, reminder_recurrence,
                    created_at, updated_at, subtasks, tags, attachments
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
                [
                    id,
                    listId,
                    note.title,
                    note.description || '',
                    note.done ? 1 : 0,
                    note.important ? 1 : 0,
                    0, // flagged
                    null, // due_date
                    null, // reminder_date
                    null, // reminder_time
                    'none', // reminder_recurrence
                    now,
                    now,
                    JSON.stringify([]), // subtasks
                    JSON.stringify([]), // tags
                    JSON.stringify([])  // attachments
                ]
            );
            imported++;
        } catch (error) {
            console.error(`❌ Error importing note "${note.title}":`, error);
        }
    }

    await db.close();
    console.log(`\n✅ Successfully imported ${imported} notes!`);
}

async function main() {
    console.log('🚀 Starting PDF import...\n');

    // Read all PDFs
    const pdfFiles = fs.readdirSync(PDF_DIR).filter(file => file.endsWith('.pdf'));

    console.log(`📂 Found ${pdfFiles.length} PDF files\n`);

    const allNotes: ParsedNote[] = [];

    for (const pdfFile of pdfFiles) {
        const pdfPath = path.join(PDF_DIR, pdfFile);
        const notes = await parsePdfToNotes(pdfPath);
        allNotes.push(...notes);
    }

    console.log(`\n📊 Total notes to import: ${allNotes.length}`);

    if (allNotes.length > 0) {
        await importNotesToDb(allNotes);
    }

    console.log('\n🎉 Import completed!');
}

main().catch(console.error);



