const sqlite3 = require('sqlite3');
const fs = require('fs');
const path = require('path');

// Function to open a specific database
const openDatabase = (dbFileName) => {
    return new Promise((resolve, reject) => {
        const dbPath = path.join(process.cwd(), dbFileName);
        const db = new sqlite3.Database(dbPath, (err) => {
            if (err) {
                console.error(`Error opening ${dbFileName}:`, err);
                reject(err);
            } else {
                console.log(`Connected to ${dbFileName} at:`, dbPath);
                resolve(db);
            }
        });
    });
};

// Function to parse CSV data
const parseCSV = (csvContent) => {
    const lines = csvContent.split('\n');
    const headers = lines[0].split(',');
    const data = [];

    for (let i = 1; i < lines.length; i++) {
        if (lines[i].trim()) {
            const values = lines[i].split(',');
            const row = {};
            headers.forEach((header, index) => {
                row[header.trim()] = values[index] ? values[index].trim() : '';
            });
            data.push(row);
        }
    }

    return data;
};

// Function to generate unique ID
const generateId = (prefix, index) => {
    return `${prefix}_${Date.now()}_${index}`;
};

// Helper function to parse dd/mm/yyyy date format and calculate registration date
const parseAndCalculateRegistrationDate = (expiryDateString) => {
    if (!expiryDateString) return new Date().toISOString().split('T')[0];

    // Parse dd/mm/yyyy format
    const parts = expiryDateString.split('/');
    if (parts.length === 3) {
        const day = parseInt(parts[0], 10);
        const month = parseInt(parts[1], 10) - 1; // JavaScript months are 0-based
        const year = parseInt(parts[2], 10);

        // Create expiry date
        const expiryDate = new Date(year, month, day);

        // Calculate registration date (1 year before expiry)
        const registrationDate = new Date(expiryDate);
        registrationDate.setFullYear(registrationDate.getFullYear() - 1);

        // Format as dd/mm/yyyy for consistency
        const regDay = String(registrationDate.getDate()).padStart(2, '0');
        const regMonth = String(registrationDate.getMonth() + 1).padStart(2, '0');
        const regYear = registrationDate.getFullYear();

        return `${regDay}/${regMonth}/${regYear}`;
    }

    return new Date().toISOString().split('T')[0];
};

// Function to convert CSV data to customers and domains
const convertCSVData = (csvData) => {
    const customers = [];
    const domains = [];

    csvData.forEach((row, index) => {
        if (row.domain && row.domain.trim()) {
            const customerId = generateId('customer', index);
            const domainId = generateId('domain', index);

            // Create customer record
            const customer = {
                id: customerId,
                name: row.owner || row.domain.split('.')[0] || 'לקוח',
                email: row.email || '',
                phone: row.phone || '',
                company: row.owner || '',
                status: row.status === 'פעיל' ? 'active' : 'lead',
                stage: 'client',
                assigned_sales_rep: 'טיטי',
                value: parseFloat(row.amount) || 0,
                notes: row.notes || '',
                created_date: row.createdAt || new Date().toISOString(),
                last_activity: row.updatedAt || new Date().toISOString(),
                updated_at: new Date().toISOString()
            };

            // Create domain record
            const domain = {
                id: domainId,
                customer_id: customerId,
                name: row.domain,
                registration_date: parseAndCalculateRegistrationDate(row.nextPayment),
                expiry_date: row.nextPayment || new Date().toISOString(),
                status: row.status || 'active',
                provider: 'TimWebs',
                price: parseFloat(row.amount) || 0,
                notes: `Package: ${row.package || 'N/A'}, Priority: ${row.priority || 'N/A'}`,
                created_at: new Date().toISOString(),
                updated_at: new Date().toISOString()
            };

            customers.push(customer);
            domains.push(domain);
        }
    });

    return { customers, domains };
};

// Main import function
const importCSVData = async () => {
    try {
        console.log('🔄 Starting CSV data import...');

        // Read CSV file
        const csvPath = path.join(process.cwd(), '.vscode/project log + md/שגיות כללי/phones_all_2025-10-07 - phones_all_2025-10-07-titi1.csv');
        const csvContent = fs.readFileSync(csvPath, 'utf8');
        console.log('✅ CSV file read successfully');

        // Parse CSV data
        const csvData = parseCSV(csvContent);
        console.log(`✅ Parsed ${csvData.length} records from CSV`);

        // Convert to customers and domains
        const { customers, domains } = convertCSVData(csvData);
        console.log(`✅ Converted to ${customers.length} customers and ${domains.length} domains`);

        // Open comprehensive database
        const db = await openDatabase('comprehensive.db');

        // Clear existing sample data
        await new Promise((resolve, reject) => {
            db.run('DELETE FROM customers', (err) => {
                if (err) reject(err); else resolve();
            });
        });
        await new Promise((resolve, reject) => {
            db.run('DELETE FROM domains', (err) => {
                if (err) reject(err); else resolve();
            });
        });
        console.log('✅ Cleared existing sample data');

        // Insert customers
        for (const customer of customers) {
            await new Promise((resolve, reject) => {
                db.run(`
                    INSERT INTO customers (id, name, email, phone, company, status, stage, assigned_sales_rep, value, notes, created_date, last_activity, updated_at)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                `, [
                    customer.id, customer.name, customer.email, customer.phone, customer.company,
                    customer.status, customer.stage, customer.assigned_sales_rep, customer.value,
                    customer.notes, customer.created_date, customer.last_activity, customer.updated_at
                ], function (err) {
                    if (err) reject(err); else resolve();
                });
            });
        }
        console.log(`✅ Inserted ${customers.length} customers`);

        // Insert domains
        for (const domain of domains) {
            await new Promise((resolve, reject) => {
                db.run(`
                    INSERT INTO domains (id, customer_id, name, registration_date, expiry_date, status, provider, price, notes, created_at, updated_at)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                `, [
                    domain.id, domain.customer_id, domain.name, domain.registration_date,
                    domain.expiry_date, domain.status, domain.provider, domain.price, domain.notes,
                    domain.created_at, domain.updated_at
                ], function (err) {
                    if (err) reject(err); else resolve();
                });
            });
        }
        console.log(`✅ Inserted ${domains.length} domains`);

        // Close database
        db.close();

        console.log('🎉 CSV data import completed successfully!');
        console.log(`📊 Summary: ${customers.length} customers and ${domains.length} domains imported`);

    } catch (error) {
        console.error('❌ Error importing CSV data:', error);
    }
};

// Run the import
importCSVData();
