const Database = require('better-sqlite3');
const mysql = require('mysql2/promise');
const fs = require('fs');
const path = require('path');

async function convertSqliteToMysql() {
    console.log('🔄 Starting SQLite to MySQL conversion...');

    // הגדרות MySQL - שנה את הפרטים שלך
    const mysqlConfig = {
        host: 'localhost',
        user: 'your_username', // שנה לשם המשתמש שלך
        password: 'your_password', // שנה לסיסמה שלך
        database: 'your_database', // שנה לשם מסד הנתונים שלך
        multipleStatements: true
    };

    try {
        // חיבור ל-MySQL
        console.log('📡 Connecting to MySQL...');
        const mysqlConnection = await mysql.createConnection(mysqlConfig);

        // חיבור ל-SQLite
        console.log('📁 Reading SQLite databases...');
        const sqliteFiles = [
            'comprehensive.db',
            'domains.db',
            'crm.db',
            'notes.db',
            'phones.db',
            'auth.db'
        ];

        for (const dbFile of sqliteFiles) {
            if (fs.existsSync(dbFile)) {
                console.log(`🔄 Converting ${dbFile}...`);
                await convertDatabase(dbFile, mysqlConnection);
            }
        }

        await mysqlConnection.end();
        console.log('✅ Conversion completed successfully!');

    } catch (error) {
        console.error('❌ Conversion failed:', error);
    }
}

async function convertDatabase(sqliteFile, mysqlConnection) {
    const sqliteDb = new Database(sqliteFile);

    try {
        // קבלת רשימת טבלאות
        const tables = sqliteDb.prepare(`
            SELECT name FROM sqlite_master 
            WHERE type='table' AND name NOT LIKE 'sqlite_%'
        `).all();

        for (const table of tables) {
            console.log(`  📋 Converting table: ${table.name}`);

            // יצירת טבלה ב-MySQL
            await createMySQLTable(table.name, sqliteDb, mysqlConnection);

            // העתקת נתונים
            await copyData(table.name, sqliteDb, mysqlConnection);
        }

    } finally {
        sqliteDb.close();
    }
}

async function createMySQLTable(tableName, sqliteDb, mysqlConnection) {
    // קבלת מבנה הטבלה
    const columns = sqliteDb.prepare(`PRAGMA table_info(${tableName})`).all();

    let createTableSQL = `CREATE TABLE IF NOT EXISTS ${tableName} (`;

    columns.forEach((col, index) => {
        let columnDef = `${col.name} `;

        // המרת סוגי נתונים
        switch (col.type.toUpperCase()) {
            case 'INTEGER':
                if (col.pk) columnDef += 'INT AUTO_INCREMENT PRIMARY KEY';
                else columnDef += 'INT';
                break;
            case 'TEXT':
                columnDef += 'TEXT';
                break;
            case 'REAL':
                columnDef += 'DECIMAL(10,2)';
                break;
            case 'BLOB':
                columnDef += 'LONGBLOB';
                break;
            default:
                columnDef += 'TEXT';
        }

        if (col.notnull && !col.pk) columnDef += ' NOT NULL';
        if (col.dflt_value !== null) columnDef += ` DEFAULT '${col.dflt_value}'`;

        createTableSQL += columnDef;
        if (index < columns.length - 1) createTableSQL += ', ';
    });

    createTableSQL += ')';

    await mysqlConnection.execute(createTableSQL);
}

async function copyData(tableName, sqliteDb, mysqlConnection) {
    const rows = sqliteDb.prepare(`SELECT * FROM ${tableName}`).all();

    if (rows.length === 0) return;

    const columns = Object.keys(rows[0]);
    const placeholders = columns.map(() => '?').join(', ');
    const insertSQL = `INSERT INTO ${tableName} (${columns.join(', ')}) VALUES (${placeholders})`;

    for (const row of rows) {
        const values = columns.map(col => row[col]);
        await mysqlConnection.execute(insertSQL, values);
    }

    console.log(`    ✅ Copied ${rows.length} rows`);
}

// הרצה
if (require.main === module) {
    convertSqliteToMysql();
}

module.exports = { convertSqliteToMysql };
