import { NextRequest, NextResponse } from 'next/server';
import { jwtVerify } from 'jose';

const JWT_SECRET = process.env.JWT_SECRET || 'your-super-secret-jwt-key-change-in-production-123456789';

interface JWTPayload {
    userId: string;
    email: string;
    role: 'ADMIN' | 'USER';
}

function getAuthToken(request: NextRequest): string | null {
    // Try to get token from cookies first
    const cookieToken = request.cookies.get('auth-token')?.value;
    console.log('🍪 Cookie token:', cookieToken ? `${cookieToken.substring(0, 20)}...` : 'none');

    if (cookieToken) {
        return cookieToken;
    }

    // Try to get token from Authorization header
    const authHeader = request.headers.get('authorization');
    console.log('🔑 Auth header:', authHeader ? `${authHeader.substring(0, 20)}...` : 'none');

    if (authHeader && authHeader.startsWith('Bearer ')) {
        return authHeader.substring(7);
    }

    return null;
}

async function verifyToken(token: string): Promise<JWTPayload | null> {
    try {
        console.log('🔐 Verifying token with secret:', JWT_SECRET ? 'exists' : 'missing');
        console.log('🔐 Token to verify:', token ? `${token.substring(0, 20)}...` : 'none');

        const secret = new TextEncoder().encode(JWT_SECRET);
        const { payload } = await jwtVerify(token, secret);

        console.log('✅ Token verification successful:', payload);
        return payload as JWTPayload;
    } catch (error) {
        console.log('❌ Token verification failed:', error);
        return null;
    }
}

function isPublicRoute(pathname: string): boolean {
    const publicRoutes = [
        '/login',
        '/login-simple',
        '/test-auth',
        '/register',
        '/forgot-password',
        '/reset-password',
        '/api/auth/login',
        '/api/auth/register',
        '/api/auth/forgot-password',
        '/api/auth/reset-password',
        '/api/auth/logout',
        '/api/health'
    ];

    return publicRoutes.some(route => pathname.startsWith(route));
}

function isAdminRoute(pathname: string): boolean {
    return pathname.startsWith('/admin');
}

export async function middleware(request: NextRequest) {
    const { pathname } = request.nextUrl;

    // Allow public routes
    if (isPublicRoute(pathname)) {
        return NextResponse.next();
    }

    // Check authentication
    const token = getAuthToken(request);
    console.log('🔍 Middleware - Path:', pathname, 'Token exists:', !!token);

    if (!token) {
        console.log('❌ No token found, redirecting to login');
        // No token, redirect to login
        const loginUrl = new URL('/login', request.url);
        loginUrl.searchParams.set('next', pathname);
        return NextResponse.redirect(loginUrl);
    }

    // Verify token
    const payload = await verifyToken(token);
    console.log('🔍 Token payload:', payload);

    if (!payload) {
        console.log('❌ Invalid token, redirecting to login');
        // Invalid token, redirect to login
        const loginUrl = new URL('/login', request.url);
        loginUrl.searchParams.set('next', pathname);
        return NextResponse.redirect(loginUrl);
    }

    console.log('✅ Token valid, allowing access');

    // Check admin routes
    if (isAdminRoute(pathname) && payload.role !== 'ADMIN') {
        // Not admin, redirect to dashboard
        return NextResponse.redirect(new URL('/', request.url));
    }

    // All good, continue
    return NextResponse.next();
}

export const config = {
    matcher: [
        /*
         * Match all request paths except for the ones starting with:
         * - api (API routes)
         * - _next/static (static files)
         * - _next/image (image optimization files)
         * - favicon.ico (favicon file)
         * - public folder
         */
        '/((?!api|_next/static|_next/image|favicon.ico|.*\\.(?:svg|png|jpg|jpeg|gif|webp)$).*)',
    ],
};
