import sqlite3 from 'sqlite3';
import { Database } from 'sqlite3';
import path from 'path';

let db: Database | null = null;

export const openDB = (): Promise<Database> => {
    return new Promise((resolve, reject) => {
        if (db) {
            resolve(db);
            return;
        }

        // Use comprehensive.db as the main database
        const dbPath = path.join(process.cwd(), 'comprehensive.db');
        const database = new sqlite3.Database(dbPath, (err) => {
            if (err) {
                console.error('Error opening database:', err);
                reject(err);
            } else {
                console.log('Connected to SQLite database at:', dbPath);
                db = database;
                resolve(database);
            }
        });
    });
};

export const closeDB = (): Promise<void> => {
    return new Promise((resolve, reject) => {
        if (db) {
            db.close((err) => {
                if (err) {
                    console.error('Error closing database:', err);
                    reject(err);
                } else {
                    console.log('Database connection closed');
                    db = null;
                    resolve();
                }
            });
        } else {
            resolve();
        }
    });
};

// Initialize database tables
export const initDB = async (): Promise<void> => {
    const database = await openDB();

    return new Promise((resolve, reject) => {
        database.serialize(() => {
            // Create recordings table
            database.run(`
                CREATE TABLE IF NOT EXISTS task_recordings (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    task_id INTEGER NOT NULL,
                    name TEXT NOT NULL,
                    duration TEXT NOT NULL,
                    audio_data BLOB NOT NULL,
                    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                    FOREIGN KEY (task_id) REFERENCES tasks (id)
                )
            `, (err) => {
                if (err) {
                    console.error('Error creating recordings table:', err);
                    reject(err);
                } else {
                    console.log('Recordings table created successfully');
                    resolve();
                }
            });
        });
    });
};

// Save recording to database
export const saveRecording = async (taskId: number, name: string, duration: string, audioBlob: Blob): Promise<number> => {
    const database = await openDB();

    return new Promise((resolve, reject) => {
        const stmt = database.prepare(`
            INSERT INTO task_recordings (task_id, name, duration, audio_data)
            VALUES (?, ?, ?, ?)
        `);

        stmt.run([taskId, name, duration, audioBlob], function (err) {
            if (err) {
                console.error('Error saving recording:', err);
                reject(err);
            } else {
                console.log('Recording saved with ID:', this.lastID);
                resolve(this.lastID);
            }
        });

        stmt.finalize();
    });
};

// Load recordings for a task
export const loadRecordings = async (taskId: number): Promise<Array<{ id: number, name: string, duration: string, audio_data: Buffer, created_at: string }>> => {
    const database = await openDB();

    return new Promise((resolve, reject) => {
        database.all(`
            SELECT id, name, duration, audio_data, created_at
            FROM task_recordings
            WHERE task_id = ?
            ORDER BY created_at DESC
        `, [taskId], (err, rows) => {
            if (err) {
                console.error('Error loading recordings:', err);
                reject(err);
            } else {
                resolve(rows as Array<{ id: number, name: string, duration: string, audio_data: Buffer, created_at: string }>);
            }
        });
    });
};

// Delete recording
export const deleteRecording = async (recordingId: number): Promise<void> => {
    const database = await openDB();

    return new Promise((resolve, reject) => {
        database.run(`
            DELETE FROM task_recordings
            WHERE id = ?
        `, [recordingId], (err) => {
            if (err) {
                console.error('Error deleting recording:', err);
                reject(err);
            } else {
                console.log('Recording deleted');
                resolve();
            }
        });
    });
};
