import sqlite3 from 'sqlite3';
import { Database } from 'sqlite3';
import path from 'path';
import bcrypt from 'bcryptjs';
import { User, CreateUserRequest } from '@/src/types/user';

let authDb: Database | null = null;

export const getAuthDatabase = (): Promise<Database> => {
    return new Promise((resolve, reject) => {
        if (authDb) {
            resolve(authDb);
            return;
        }

        const dbPath = path.join(process.cwd(), 'auth.db');
        const database = new sqlite3.Database(dbPath, (err) => {
            if (err) {
                console.error('Error opening auth database:', err);
                reject(err);
            } else {
                console.log('Connected to auth SQLite database at:', dbPath);
                authDb = database;
                resolve(database);
            }
        });
    });
};

export const initAuthDatabase = async (): Promise<void> => {
    const database = await getAuthDatabase();

    return new Promise((resolve, reject) => {
        const createUsersTableSQL = `
            CREATE TABLE IF NOT EXISTS users (
                id TEXT PRIMARY KEY,
                email TEXT UNIQUE NOT NULL,
                password TEXT NOT NULL,
                name TEXT NOT NULL,
                role TEXT NOT NULL CHECK (role IN ('ADMIN', 'USER')),
                isActive BOOLEAN DEFAULT 1,
                createdAt TEXT NOT NULL,
                updatedAt TEXT NOT NULL,
                lastLoginAt TEXT
            )
        `;

        const createPasswordResetTableSQL = `
            CREATE TABLE IF NOT EXISTS password_reset_tokens (
                id TEXT PRIMARY KEY,
                userId TEXT NOT NULL,
                token TEXT UNIQUE NOT NULL,
                expiresAt TEXT NOT NULL,
                createdAt TEXT NOT NULL,
                FOREIGN KEY (userId) REFERENCES users (id) ON DELETE CASCADE
            )
        `;

        database.run(createUsersTableSQL, (err) => {
            if (err) {
                console.error('Error creating users table:', err);
                reject(err);
                return;
            }

            database.run(createPasswordResetTableSQL, (err) => {
                if (err) {
                    console.error('Error creating password_reset_tokens table:', err);
                    reject(err);
                    return;
                }

                // Create default admin user if no users exist
                database.get('SELECT COUNT(*) as count FROM users', (err, row: any) => {
                    if (err) {
                        console.error('Error checking users count:', err);
                        reject(err);
                        return;
                    }

                    if (row.count === 0) {
                        createDefaultAdminUser(database).then(() => {
                            resolve();
                        }).catch(reject);
                    } else {
                        resolve();
                    }
                });
            });
        });
    });
};

const createDefaultAdminUser = async (database: Database): Promise<void> => {
    const hashedPassword = await bcrypt.hash('admin123', 12);
    const now = new Date().toISOString();
    const userId = `admin_${Date.now()}`;

    return new Promise((resolve, reject) => {
        database.run(
            `INSERT INTO users (id, email, password, name, role, isActive, createdAt, updatedAt) 
             VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
            [userId, 'admin@domain-manager.com', hashedPassword, 'מנהל מערכת', 'ADMIN', 1, now, now],
            (err) => {
                if (err) {
                    console.error('Error creating default admin user:', err);
                    reject(err);
                } else {
                    console.log('✅ Default admin user created: admin@domain-manager.com / admin123');
                    resolve();
                }
            }
        );
    });
};

export const createUser = async (userData: CreateUserRequest): Promise<User> => {
    const database = await getAuthDatabase();
    const hashedPassword = await bcrypt.hash(userData.password, 12);
    const now = new Date().toISOString();
    const userId = `user_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;

    return new Promise((resolve, reject) => {
        database.run(
            `INSERT INTO users (id, email, password, name, role, isActive, createdAt, updatedAt) 
             VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
            [
                userId,
                userData.email,
                hashedPassword,
                userData.name,
                userData.role || 'USER',
                1,
                now,
                now
            ],
            function (err) {
                if (err) {
                    reject(err);
                } else {
                    resolve({
                        id: userId,
                        email: userData.email,
                        password: hashedPassword,
                        name: userData.name,
                        role: (userData.role || 'USER') as 'ADMIN' | 'USER',
                        isActive: true,
                        createdAt: now,
                        updatedAt: now
                    });
                }
            }
        );
    });
};

export const findUserByEmail = async (email: string): Promise<User | null> => {
    const database = await getAuthDatabase();

    return new Promise((resolve, reject) => {
        database.get(
            'SELECT * FROM users WHERE email = ? AND isActive = 1',
            [email],
            (err, row: any) => {
                if (err) {
                    reject(err);
                } else {
                    resolve(row || null);
                }
            }
        );
    });
};

export const findUserById = async (id: string): Promise<Omit<User, 'password'> | null> => {
    const database = await getAuthDatabase();

    return new Promise((resolve, reject) => {
        database.get(
            'SELECT id, email, name, role, isActive, createdAt, updatedAt, lastLoginAt FROM users WHERE id = ? AND isActive = 1',
            [id],
            (err, row: any) => {
                if (err) {
                    reject(err);
                } else {
                    resolve(row || null);
                }
            }
        );
    });
};

export const updateLastLogin = async (userId: string): Promise<void> => {
    const database = await getAuthDatabase();
    const now = new Date().toISOString();

    return new Promise((resolve, reject) => {
        database.run(
            'UPDATE users SET lastLoginAt = ?, updatedAt = ? WHERE id = ?',
            [now, now, userId],
            (err) => {
                if (err) {
                    reject(err);
                } else {
                    resolve();
                }
            }
        );
    });
};

export const createPasswordResetToken = async (userId: string, token: string): Promise<void> => {
    const database = await getAuthDatabase();
    const expiresAt = new Date(Date.now() + 3600000).toISOString(); // 1 hour
    const now = new Date().toISOString();
    const tokenId = `token_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;

    return new Promise((resolve, reject) => {
        database.run(
            `INSERT INTO password_reset_tokens (id, userId, token, expiresAt, createdAt) 
             VALUES (?, ?, ?, ?, ?)`,
            [tokenId, userId, token, expiresAt, now],
            (err) => {
                if (err) {
                    reject(err);
                } else {
                    resolve();
                }
            }
        );
    });
};

export const validatePasswordResetToken = async (token: string): Promise<{ userId: string } | null> => {
    const database = await getAuthDatabase();
    const now = new Date().toISOString();

    return new Promise((resolve, reject) => {
        database.get(
            'SELECT userId FROM password_reset_tokens WHERE token = ? AND expiresAt > ?',
            [token, now],
            (err, row: any) => {
                if (err) {
                    reject(err);
                } else {
                    resolve(row ? { userId: row.userId } : null);
                }
            }
        );
    });
};

export const updatePassword = async (userId: string, newPassword: string): Promise<void> => {
    const database = await getAuthDatabase();
    const hashedPassword = await bcrypt.hash(newPassword, 12);
    const now = new Date().toISOString();

    return new Promise((resolve, reject) => {
        database.run(
            'UPDATE users SET password = ?, updatedAt = ? WHERE id = ?',
            [hashedPassword, now, userId],
            (err) => {
                if (err) {
                    reject(err);
                } else {
                    resolve();
                }
            }
        );
    });
};

export const deletePasswordResetToken = async (token: string): Promise<void> => {
    const database = await getAuthDatabase();

    return new Promise((resolve, reject) => {
        database.run(
            'DELETE FROM password_reset_tokens WHERE token = ?',
            [token],
            (err) => {
                if (err) {
                    reject(err);
                } else {
                    resolve();
                }
            }
        );
    });
};
