'use client';

import { useState, useRef, useEffect } from 'react';
import {
    Bold,
    Italic,
    Underline,
    List,
    ListOrdered,
    AlignRight,
    AlignLeft,
    AlignCenter,
    AlignJustify,
    Type,
    Palette,
    Table,
    Wand2,
    Maximize,
    Code
} from 'lucide-react';

interface RichTextEditorProps {
    value?: string;
    onChange: (value: string) => void;
    placeholder?: string;
    className?: string;
}

export function RichTextEditor({
    value = '',
    onChange,
    placeholder = "הקלד תוכן...",
    className
}: RichTextEditorProps) {
    const [isFocused, setIsFocused] = useState(false);
    const [isFullscreen, setIsFullscreen] = useState(false);
    const [isSourceMode, setIsSourceMode] = useState(false);
    const editorRef = useRef<HTMLDivElement>(null);
    const textareaRef = useRef<HTMLTextAreaElement>(null);

    useEffect(() => {
        if (editorRef.current) {
            editorRef.current.innerHTML = value;
        }
    }, [value]);

    const execCommand = (command: string, value?: string) => {
        if (isSourceMode) return; // Don't execute commands in source mode

        // Focus the editor before executing commands
        if (editorRef.current) {
            editorRef.current.focus();
            document.execCommand(command, false, value);
            onChange(editorRef.current.innerHTML);
        }
    };

    const handleInput = () => {
        if (editorRef.current) {
            onChange(editorRef.current.innerHTML);
        }
    };

    const toggleFullscreen = () => {
        setIsFullscreen(!isFullscreen);
    };

    const toggleSourceMode = () => {
        if (isSourceMode) {
            // Switch from source to visual
            if (textareaRef.current) {
                const htmlContent = textareaRef.current.value;
                onChange(htmlContent);
                if (editorRef.current) {
                    editorRef.current.innerHTML = htmlContent;
                }
            }
        } else {
            // Switch from visual to source
            if (editorRef.current) {
                onChange(editorRef.current.innerHTML);
            }
        }
        setIsSourceMode(!isSourceMode);
    };

    const insertTable = () => {
        const rows = prompt('כמה שורות?', '3');
        const cols = prompt('כמה עמודות?', '3');
        if (rows && cols) {
            let tableHTML = '<table border="1" style="border-collapse: collapse; width: 100%;">';
            for (let i = 0; i < parseInt(rows); i++) {
                tableHTML += '<tr>';
                for (let j = 0; j < parseInt(cols); j++) {
                    tableHTML += '<td style="border: 1px solid #ddd; padding: 8px;">&nbsp;</td>';
                }
                tableHTML += '</tr>';
            }
            tableHTML += '</table>';
            execCommand('insertHTML', tableHTML);
        }
    };

    const setAlignment = (align: string) => {
        if (isSourceMode) return;

        if (editorRef.current) {
            editorRef.current.focus();

            // Use execCommand with formatBlock to ensure proper alignment
            const selection = window.getSelection();
            if (selection && selection.rangeCount > 0) {
                const range = selection.getRangeAt(0);

                // Create a div with the alignment
                const div = document.createElement('div');
                div.style.textAlign = align;
                div.style.width = '100%';

                try {
                    range.surroundContents(div);
                } catch (e) {
                    // If surroundContents fails, try a different approach
                    const contents = range.extractContents();
                    div.appendChild(contents);
                    range.insertNode(div);
                }

                onChange(editorRef.current.innerHTML);
            } else {
                // If no selection, apply to the entire content
                editorRef.current.style.textAlign = align;
                onChange(editorRef.current.innerHTML);
            }
        }
    };

    const ToolbarButton = ({
        onClick,
        children,
        title,
        isActive = false
    }: {
        onClick: () => void;
        children: React.ReactNode;
        title: string;
        isActive?: boolean;
    }) => (
        <button
            type="button"
            onClick={onClick}
            title={title}
            className={`p-2 rounded hover:bg-gray-100 transition-colors ${isActive ? 'bg-blue-100 text-blue-600' : 'text-gray-600'
                }`}
        >
            {children}
        </button>
    );

    const containerClasses = isFullscreen
        ? 'fixed inset-0 z-50 bg-white'
        : `relative border border-gray-300 rounded-md ${className}`;

    return (
        <div className={containerClasses}>
            {/* Toolbar */}
            <div className="border-b border-gray-200 p-2 flex items-center gap-1 flex-wrap">
                {/* תצוגה */}
                <div className="flex items-center gap-1 mr-4">
                    <ToolbarButton
                        onClick={toggleSourceMode}
                        title="קוד מקור"
                        isActive={isSourceMode}
                    >
                        <Code className="w-4 h-4" />
                    </ToolbarButton>
                    <ToolbarButton onClick={toggleFullscreen} title="מסך מלא">
                        <Maximize className="w-4 h-4" />
                    </ToolbarButton>
                </div>

                {/* עיצוב טקסט */}
                <div className="flex items-center gap-1 mr-4">
                    <ToolbarButton onClick={() => execCommand('bold')} title="הדגשה">
                        <Bold className="w-4 h-4" />
                    </ToolbarButton>
                    <ToolbarButton onClick={() => execCommand('italic')} title="הטיה">
                        <Italic className="w-4 h-4" />
                    </ToolbarButton>
                    <ToolbarButton onClick={() => execCommand('underline')} title="קו תחתון">
                        <Underline className="w-4 h-4" />
                    </ToolbarButton>
                    <input
                        type="color"
                        onChange={(e) => execCommand('foreColor', e.target.value)}
                        className="w-8 h-8 rounded cursor-pointer border border-gray-300"
                        title="צבע טקסט"
                    />
                </div>

                {/* גופן */}
                <div className="flex items-center gap-1 mr-4">
                    <select
                        className="text-sm border border-gray-300 rounded px-2 py-1"
                        onChange={(e) => execCommand('fontName', e.target.value)}
                        title="בחר גופן"
                        aria-label="בחר גופן"
                    >
                        <option value="">גופן</option>
                        <option value="Arial">Arial</option>
                        <option value="Open Sans">Open Sans</option>
                        <option value="Times New Roman">Times New Roman</option>
                        <option value="Helvetica">Helvetica</option>
                    </select>
                    <select
                        className="text-sm border border-gray-300 rounded px-2 py-1"
                        onChange={(e) => execCommand('fontSize', e.target.value)}
                        title="גודל גופן"
                        aria-label="גודל גופן"
                    >
                        <option value="">גודל</option>
                        <option value="1">קטן מאוד</option>
                        <option value="2">קטן</option>
                        <option value="3">בינוני</option>
                        <option value="4">גדול</option>
                        <option value="5">גדול מאוד</option>
                        <option value="6">ענק</option>
                        <option value="7">ענקי</option>
                    </select>
                </div>

                {/* יישור */}
                <div className="flex items-center gap-1 mr-4">
                    <ToolbarButton onClick={() => {
                        if (editorRef.current) {
                            editorRef.current.focus();
                            setTimeout(() => {
                                document.execCommand('justifyLeft', false);
                                onChange(editorRef.current?.innerHTML || '');
                            }, 10);
                        }
                    }} title="יישור לשמאל">
                        <AlignLeft className="w-4 h-4" />
                    </ToolbarButton>
                    <ToolbarButton onClick={() => {
                        if (editorRef.current) {
                            editorRef.current.focus();
                            setTimeout(() => {
                                document.execCommand('justifyCenter', false);
                                onChange(editorRef.current?.innerHTML || '');
                            }, 10);
                        }
                    }} title="יישור למרכז">
                        <AlignCenter className="w-4 h-4" />
                    </ToolbarButton>
                    <ToolbarButton onClick={() => {
                        if (editorRef.current) {
                            editorRef.current.focus();
                            setTimeout(() => {
                                document.execCommand('justifyRight', false);
                                onChange(editorRef.current?.innerHTML || '');
                            }, 10);
                        }
                    }} title="יישור לימין">
                        <AlignRight className="w-4 h-4" />
                    </ToolbarButton>
                    <ToolbarButton onClick={() => {
                        if (editorRef.current) {
                            editorRef.current.focus();
                            setTimeout(() => {
                                document.execCommand('justifyFull', false);
                                onChange(editorRef.current?.innerHTML || '');
                            }, 10);
                        }
                    }} title="יישור לשני הצדדים">
                        <AlignJustify className="w-4 h-4" />
                    </ToolbarButton>
                </div>

                {/* רשימות */}
                <div className="flex items-center gap-1 mr-4">
                    <ToolbarButton onClick={() => execCommand('insertUnorderedList')} title="רשימה לא ממוספרת">
                        <List className="w-4 h-4" />
                    </ToolbarButton>
                    <ToolbarButton onClick={() => execCommand('insertOrderedList')} title="רשימה ממוספרת">
                        <ListOrdered className="w-4 h-4" />
                    </ToolbarButton>
                </div>

                {/* כלים נוספים */}
                <div className="flex items-center gap-1">
                    <ToolbarButton onClick={() => execCommand('indent')} title="הזחה">
                        <Type className="w-4 h-4" />
                    </ToolbarButton>
                    <ToolbarButton onClick={insertTable} title="טבלה">
                        <Table className="w-4 h-4" />
                    </ToolbarButton>
                    <ToolbarButton onClick={() => execCommand('insertHorizontalRule')} title="קו אופקי">
                        <Wand2 className="w-4 h-4" />
                    </ToolbarButton>
                </div>
            </div>

            {/* Editor Area */}
            {isSourceMode ? (
                <textarea
                    ref={textareaRef}
                    value={value}
                    onChange={(e) => onChange(e.target.value)}
                    onFocus={() => setIsFocused(true)}
                    onBlur={() => setIsFocused(false)}
                    className={`min-h-[200px] p-4 focus:outline-none w-full font-mono text-sm border-0 ${isFocused ? 'ring-2 ring-blue-500' : ''
                        }`}
                    style={{
                        minHeight: isFullscreen ? 'calc(100vh - 120px)' : '200px',
                        resize: 'none'
                    }}
                    dir="ltr"
                    placeholder={placeholder}
                />
            ) : (
                <div
                    ref={editorRef}
                    contentEditable
                    onInput={handleInput}
                    onFocus={() => setIsFocused(true)}
                    onBlur={() => setIsFocused(false)}
                    className={`min-h-[200px] p-4 focus:outline-none ${isFocused ? 'ring-2 ring-blue-500' : ''
                        }`}
                    style={{
                        minHeight: isFullscreen ? 'calc(100vh - 120px)' : '200px'
                    }}
                    dir="rtl"
                    data-placeholder={placeholder}
                    suppressContentEditableWarning={true}
                />
            )}

            {/* ספירת תווים */}
            <div className="absolute bottom-2 left-2 text-xs text-gray-500 bg-white px-2 py-1 rounded shadow-sm">
                {value.replace(/<[^>]*>/g, '').length} תווים
            </div>
        </div>
    );
}
