'use client';

import React, { useEffect, useMemo, useState } from 'react';
import { Pencil, Trash2 } from 'lucide-react';
import TaskDetailsModal from './TaskDetailsModal';

type TaskStatus = 'pending' | 'in_progress' | 'completed' | 'cancelled' | 'on_hold';
type TaskPriority = 'low' | 'medium' | 'high' | 'urgent';

interface TaskRow {
    id: number;
    title: string;
    description?: string;
    status: TaskStatus;
    priority: TaskPriority;
    due_date?: string;
    assigned_to?: string;
    progress: number;
    category?: string;
    customer_name?: string;
    domain?: string;
    project_title?: string;
}

const statusOptions: { value: TaskStatus | 'all'; label: string }[] = [
    { value: 'all', label: 'כל הסטטוסים' },
    { value: 'pending', label: 'ממתין' },
    { value: 'in_progress', label: 'בתהליך' },
    { value: 'completed', label: 'הושלם' },
    { value: 'on_hold', label: 'בהמתנה' },
    { value: 'cancelled', label: 'בוטל' },
];

const priorityOptions: { value: TaskPriority | 'all'; label: string }[] = [
    { value: 'all', label: 'כל העדיפויות' },
    { value: 'urgent', label: 'דחוף' },
    { value: 'high', label: 'גבוה' },
    { value: 'medium', label: 'בינוני' },
    { value: 'low', label: 'נמוך' },
];

const pageSizeOptions = [10, 50, 100];

// קבוצות קטגוריות
const categoryGroups = [
    {
        label: '👥 ניהול לקוחות ומכירות',
        categories: [
            { value: 'customer', label: 'לקוחות', icon: '👥' },
            { value: 'sales', label: 'מכירות', icon: '💼' },
            { value: 'finance', label: 'כספים', icon: '💰' },
        ]
    },
    {
        label: '🌐 דומיינים ואימייל',
        categories: [
            { value: 'domain', label: 'דומיינים', icon: '🌐' },
            { value: 'email', label: 'אימייל', icon: '📧' },
        ]
    },
    {
        label: '💻 טכנולוגיה ופיתוח',
        categories: [
            { value: 'technical', label: 'טכני', icon: '💻' },
            { value: 'development', label: 'פיתוח', icon: '🎨' },
            { value: 'maintenance', label: 'תחזוקה', icon: '🔧' },
        ]
    },
    {
        label: '🔒 אבטחה ואיכות',
        categories: [
            { value: 'security', label: 'אבטחה', icon: '🔒' },
            { value: 'quality', label: 'בקרת איכות', icon: '✅' },
        ]
    },
    {
        label: '📊 ניהול ותפעול',
        categories: [
            { value: 'project', label: 'פרויקטים', icon: '📊' },
            { value: 'operations', label: 'תפעול', icon: '⚙️' },
            { value: 'administrative', label: 'מנהלתי', icon: '📝' },
        ]
    },
    {
        label: '📢 שיווק ותמיכה',
        categories: [
            { value: 'marketing', label: 'שיווק', icon: '📢' },
            { value: 'support', label: 'תמיכה', icon: '🎓' },
        ]
    },
    {
        label: '🎓 מחקר והדרכה',
        categories: [
            { value: 'research', label: 'מחקר', icon: '🔬' },
            { value: 'training', label: 'הדרכה פנימית', icon: '🎓' },
            { value: 'legal', label: 'משפטי', icon: '⚖️' },
        ]
    },
    {
        label: '📌 אחר',
        categories: [
            { value: 'other', label: 'אחר', icon: '📌' },
        ]
    }
];

// פונקציה עזר לקבלת שם קטגוריה עם אייקון
function getCategoryLabel(categoryValue: string): string {
    for (const group of categoryGroups) {
        const category = group.categories.find(cat => cat.value === categoryValue);
        if (category) {
            return `${category.icon} ${category.label}`;
        }
    }
    return categoryValue;
}

function chipClassByStatus(status: TaskStatus): string {
    switch (status) {
        case 'completed':
            return 'bg-green-100 text-green-800 border-green-200';
        case 'in_progress':
            return 'bg-blue-100 text-blue-800 border-blue-200';
        case 'pending':
            return 'bg-yellow-100 text-yellow-800 border-yellow-200';
        case 'on_hold':
            return 'bg-gray-100 text-gray-800 border-gray-200';
        case 'cancelled':
            return 'bg-red-100 text-red-800 border-red-200';
        default:
            return 'bg-gray-100 text-gray-800 border-gray-200';
    }
}

function chipClassByPriority(priority: TaskPriority): string {
    switch (priority) {
        case 'urgent':
            return 'bg-red-100 text-red-800 border-red-200';
        case 'high':
            return 'bg-orange-100 text-orange-800 border-orange-200';
        case 'medium':
            return 'bg-indigo-100 text-indigo-800 border-indigo-200';
        case 'low':
            return 'bg-gray-100 text-gray-800 border-gray-200';
        default:
            return 'bg-gray-100 text-gray-800 border-gray-200';
    }
}

export default function TasksTable() {
    const [tasks, setTasks] = useState<TaskRow[]>([]);
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState<string | null>(null);

    const [status, setStatus] = useState<TaskStatus | 'all'>('all');
    const [priority, setPriority] = useState<TaskPriority | 'all'>('all');
    const [query, setQuery] = useState('');

    const [pageSize, setPageSize] = useState<number>(10);
    const [page, setPage] = useState<number>(1);

    // Modal state
    const [selectedTask, setSelectedTask] = useState<TaskRow | null>(null);
    const [isModalOpen, setIsModalOpen] = useState(false);

    useEffect(() => {
        async function load() {
            setLoading(true);
            setError(null);
            try {
                const params = new URLSearchParams();
                if (status !== 'all') params.set('status', status);
                if (priority !== 'all') params.set('priority', priority);
                if (query.trim()) params.set('q', query.trim());
                const res = await fetch(`/api/tasks?${params.toString()}`, { cache: 'no-store' });
                if (!res.ok) throw new Error('failed');
                const data = await res.json();
                setTasks(data);
                setPage(1);
            } catch (e) {
                setError('שגיאה בטעינת משימות');
            } finally {
                setLoading(false);
            }
        }
        load();
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [status, priority, query]);

    const paged = useMemo(() => {
        const start = (page - 1) * pageSize;
        return tasks.slice(start, start + pageSize);
    }, [tasks, page, pageSize]);

    const totalPages = Math.max(1, Math.ceil(tasks.length / pageSize));

    const handleDelete = async (id: number) => {
        if (!confirm('למחוק משימה?')) return;
        const res = await fetch(`/api/tasks/${id}`, { method: 'DELETE' });
        if (res.ok) {
            setTasks(prev => prev.filter(t => t.id !== id));
        }
    };

    const handleTaskClick = (task: TaskRow) => {
        setSelectedTask(task);
        setIsModalOpen(true);
    };

    const handleCloseModal = () => {
        setIsModalOpen(false);
        setSelectedTask(null);
    };

    return (
        <div className="space-y-4">
            <div className="flex flex-col md:flex-row gap-3 items-stretch md:items-center">
                <div className="flex-1">
                    <input
                        value={query}
                        onChange={e => setQuery(e.target.value)}
                        placeholder="חיפוש..."
                        className="w-full border rounded px-3 py-2"
                        dir="rtl"
                    />
                </div>
                <select className="border rounded px-3 py-2" value={status} onChange={e => setStatus(e.target.value as TaskStatus | 'all')} aria-label="סינון לפי סטטוס">
                    {statusOptions.map(o => (
                        <option key={o.value} value={o.value}>{o.label}</option>
                    ))}
                </select>
                <select className="border rounded px-3 py-2" value={priority} onChange={e => setPriority(e.target.value as TaskPriority | 'all')} aria-label="סינון לפי עדיפות">
                    {priorityOptions.map(o => (
                        <option key={o.value} value={o.value}>{o.label}</option>
                    ))}
                </select>
                <select className="border rounded px-3 py-2" value={pageSize} onChange={e => setPageSize(Number(e.target.value))} aria-label="מספר פריטים בעמוד">
                    {pageSizeOptions.map(n => (
                        <option key={n} value={n}>{n} בעמוד</option>
                    ))}
                </select>
            </div>

            <div className="bg-white shadow rounded-lg overflow-hidden">
                <table className="min-w-full divide-y divide-gray-200" dir="rtl">
                    <thead className="bg-gray-50">
                        <tr>
                            <th className="px-4 py-3 text-right text-xs font-medium text-gray-500">כותרת</th>
                            <th className="px-4 py-3 text-right text-xs font-medium text-gray-500">לקוח/דומיין/פרויקט</th>
                            <th className="px-4 py-3 text-right text-xs font-medium text-gray-500">קטגוריה</th>
                            <th className="px-4 py-3 text-right text-xs font-medium text-gray-500">סטטוס</th>
                            <th className="px-4 py-3 text-right text-xs font-medium text-gray-500">עדיפות</th>
                            <th className="px-4 py-3 text-right text-xs font-medium text-gray-500">התקדמות</th>
                            <th className="px-4 py-3 text-right text-xs font-medium text-gray-500">תאריך יעד</th>
                            <th className="px-4 py-3 text-right text-xs font-medium text-gray-500">אחראי</th>
                            <th className="px-4 py-3 text-right text-xs font-medium text-gray-500">פעולות</th>
                        </tr>
                    </thead>
                    <tbody className="divide-y divide-gray-100 bg-white">
                        {loading && (
                            <tr><td className="px-4 py-4" colSpan={9}>טוען…</td></tr>
                        )}
                        {error && !loading && (
                            <tr><td className="px-4 py-4 text-red-600" colSpan={9}>{error}</td></tr>
                        )}
                        {!loading && !error && paged.length === 0 && (
                            <tr><td className="px-4 py-8 text-gray-500" colSpan={9}>אין נתונים</td></tr>
                        )}
                        {!loading && !error && paged.map(t => {
                            const relation = t.customer_name || t.domain || t.project_title || '';
                            const due = t.due_date ? new Date(t.due_date) : null;
                            const overdue = due ? due < new Date() && t.status !== 'completed' : false;
                            return (
                                <tr key={t.id} className="hover:bg-gray-50">
                                    <td className="px-4 py-3 font-medium text-gray-900">
                                        <button
                                            onClick={() => handleTaskClick(t)}
                                            className="text-blue-600 hover:text-blue-800 hover:underline cursor-pointer"
                                        >
                                            {t.title}
                                        </button>
                                    </td>
                                    <td className="px-4 py-3 text-gray-700">{relation}</td>
                                    <td className="px-4 py-3 text-gray-700">
                                        {t.category ? (
                                            <span className="inline-flex items-center px-2 py-1 rounded-full text-xs bg-blue-100 text-blue-800">
                                                {getCategoryLabel(t.category)}
                                            </span>
                                        ) : (
                                            <span className="text-gray-400">-</span>
                                        )}
                                    </td>
                                    <td className="px-4 py-3">
                                        <span className={`inline-flex items-center px-2 py-1 rounded border text-xs ${chipClassByStatus(t.status)}`}>
                                            {t.status === 'pending' && 'ממתין'}
                                            {t.status === 'in_progress' && 'בתהליך'}
                                            {t.status === 'completed' && 'הושלם'}
                                            {t.status === 'on_hold' && 'בהמתנה'}
                                            {t.status === 'cancelled' && 'בוטל'}
                                        </span>
                                    </td>
                                    <td className="px-4 py-3">
                                        <span className={`inline-flex items-center px-2 py-1 rounded border text-xs ${chipClassByPriority(t.priority)}`}>
                                            {t.priority === 'urgent' && 'דחוף'}
                                            {t.priority === 'high' && 'גבוה'}
                                            {t.priority === 'medium' && 'בינוני'}
                                            {t.priority === 'low' && 'נמוך'}
                                        </span>
                                    </td>
                                    <td className="px-4 py-3 w-48">
                                        <div className="w-full bg-gray-100 rounded h-2">
                                            <div
                                                className={`h-2 rounded ${t.progress >= 100 ? 'bg-green-500' : 'bg-blue-500'}`}
                                                style={{ width: `${Math.min(100, Math.max(0, t.progress))}%` }}
                                            />
                                        </div>
                                    </td>
                                    <td className={`px-4 py-3 ${overdue ? 'text-red-600 font-semibold' : 'text-gray-700'}`}>{t.due_date ? new Date(t.due_date).toLocaleDateString('he-IL') : '-'}</td>
                                    <td className="px-4 py-3 text-gray-700">{t.assigned_to || '-'}</td>
                                    <td className="px-4 py-3">
                                        <div className="flex gap-4 justify-start items-center">
                                            <button
                                                onClick={() => handleDelete(t.id)}
                                                className="text-red-600 hover:text-red-700"
                                                aria-label="מחיקה"
                                                title="מחיקה"
                                            >
                                                <Trash2 size={20} />
                                            </button>
                                            <a
                                                href={`/tasks/${t.id}/edit`}
                                                className="text-blue-600 hover:text-blue-700"
                                                aria-label="עריכה"
                                                title="עריכה"
                                            >
                                                <Pencil size={20} />
                                            </a>
                                        </div>
                                    </td>
                                </tr>
                            );
                        })}
                    </tbody>
                </table>
            </div>

            <div className="flex items-center justify-between">
                <div className="text-sm text-gray-600">סה"כ {tasks.length} משימות</div>
                <div className="flex items-center gap-2">
                    <button
                        className="px-3 py-1 border rounded disabled:opacity-50"
                        onClick={() => setPage(p => Math.max(1, p - 1))}
                        disabled={page <= 1}
                    >קודם</button>
                    <span className="text-sm">עמוד {page} מתוך {totalPages}</span>
                    <button
                        className="px-3 py-1 border rounded disabled:opacity-50"
                        onClick={() => setPage(p => Math.min(totalPages, p + 1))}
                        disabled={page >= totalPages}
                    >הבא</button>
                </div>
            </div>

            {/* Task Details Modal */}
            <TaskDetailsModal
                isOpen={isModalOpen}
                onClose={handleCloseModal}
                task={selectedTask}
            />
        </div>
    );
}


