'use client';

import React, { useEffect, useState } from 'react';
import { CheckCircle, Clock, AlertCircle, Pause, XCircle, TrendingUp, Calendar, Users } from 'lucide-react';

interface TasksStats {
    total: number;
    pending: number;
    in_progress: number;
    completed: number;
    on_hold: number;
    cancelled: number;
    urgent: number;
    high: number;
    medium: number;
    low: number;
    overdue: number;
    due_today: number;
    due_this_week: number;
}

export default function TasksStats() {
    const [stats, setStats] = useState<TasksStats>({
        total: 0,
        pending: 0,
        in_progress: 0,
        completed: 0,
        on_hold: 0,
        cancelled: 0,
        urgent: 0,
        high: 0,
        medium: 0,
        low: 0,
        overdue: 0,
        due_today: 0,
        due_this_week: 0,
    });
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        const fetchStats = async () => {
            try {
                const response = await fetch('/api/tasks/stats');
                if (response.ok) {
                    const data = await response.json();
                    setStats(data);
                }
            } catch (error) {
                console.error('Error fetching tasks stats:', error);
            } finally {
                setLoading(false);
            }
        };

        fetchStats();
    }, []);

    if (loading) {
        return (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                {[...Array(8)].map((_, i) => (
                    <div key={i} className="bg-white rounded-lg shadow-sm border p-6 animate-pulse">
                        <div className="flex items-center justify-between">
                            <div className="flex-1">
                                <div className="h-4 bg-gray-200 rounded mb-2"></div>
                                <div className="h-8 bg-gray-200 rounded mb-1"></div>
                                <div className="h-3 bg-gray-200 rounded"></div>
                            </div>
                            <div className="w-12 h-12 bg-gray-200 rounded-full"></div>
                        </div>
                    </div>
                ))}
            </div>
        );
    }

    const completionRate = stats.total > 0 ? Math.round((stats.completed / stats.total) * 100) : 0;
    const urgentRate = stats.total > 0 ? Math.round((stats.urgent / stats.total) * 100) : 0;

    return (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
            {/* סה"כ משימות */}
            <div className="bg-white rounded-lg shadow-sm border p-6 hover:shadow-md transition-shadow">
                <div className="flex items-center justify-between">
                    <div className="flex-1">
                        <p className="text-sm font-medium text-gray-600 mb-1">סה"כ משימות</p>
                        <p className="text-3xl font-bold text-gray-900">{stats.total}</p>
                        <p className="text-xs text-gray-500 mt-1">כל המשימות במערכת</p>
                    </div>
                    <div className="w-12 h-12 rounded-full bg-blue-500 flex items-center justify-center">
                        <CheckCircle className="w-6 h-6 text-white" />
                    </div>
                </div>
            </div>

            {/* הושלמו */}
            <div className="bg-white rounded-lg shadow-sm border p-6 hover:shadow-md transition-shadow">
                <div className="flex items-center justify-between">
                    <div className="flex-1">
                        <p className="text-sm font-medium text-gray-600 mb-1">הושלמו</p>
                        <p className="text-3xl font-bold text-green-600">{stats.completed}</p>
                        <p className="text-xs text-gray-500 mt-1">{completionRate}% מהמשימות</p>
                    </div>
                    <div className="w-12 h-12 rounded-full bg-green-500 flex items-center justify-center">
                        <CheckCircle className="w-6 h-6 text-white" />
                    </div>
                </div>
            </div>

            {/* בתהליך */}
            <div className="bg-white rounded-lg shadow-sm border p-6 hover:shadow-md transition-shadow">
                <div className="flex items-center justify-between">
                    <div className="flex-1">
                        <p className="text-sm font-medium text-gray-600 mb-1">בתהליך</p>
                        <p className="text-3xl font-bold text-blue-600">{stats.in_progress}</p>
                        <p className="text-xs text-gray-500 mt-1">משימות פעילות</p>
                    </div>
                    <div className="w-12 h-12 rounded-full bg-blue-500 flex items-center justify-center">
                        <TrendingUp className="w-6 h-6 text-white" />
                    </div>
                </div>
            </div>

            {/* ממתין */}
            <div className="bg-white rounded-lg shadow-sm border p-6 hover:shadow-md transition-shadow">
                <div className="flex items-center justify-between">
                    <div className="flex-1">
                        <p className="text-sm font-medium text-gray-600 mb-1">ממתין</p>
                        <p className="text-3xl font-bold text-yellow-600">{stats.pending}</p>
                        <p className="text-xs text-gray-500 mt-1">משימות ממתינות</p>
                    </div>
                    <div className="w-12 h-12 rounded-full bg-yellow-500 flex items-center justify-center">
                        <Clock className="w-6 h-6 text-white" />
                    </div>
                </div>
            </div>

            {/* דחוף */}
            <div className="bg-white rounded-lg shadow-sm border p-6 hover:shadow-md transition-shadow">
                <div className="flex items-center justify-between">
                    <div className="flex-1">
                        <p className="text-sm font-medium text-gray-600 mb-1">דחוף</p>
                        <p className="text-3xl font-bold text-red-600">{stats.urgent}</p>
                        <p className="text-xs text-gray-500 mt-1">{urgentRate}% מהמשימות</p>
                    </div>
                    <div className="w-12 h-12 rounded-full bg-red-500 flex items-center justify-center">
                        <AlertCircle className="w-6 h-6 text-white" />
                    </div>
                </div>
            </div>

            {/* בהמתנה */}
            <div className="bg-white rounded-lg shadow-sm border p-6 hover:shadow-md transition-shadow">
                <div className="flex items-center justify-between">
                    <div className="flex-1">
                        <p className="text-sm font-medium text-gray-600 mb-1">בהמתנה</p>
                        <p className="text-3xl font-bold text-purple-600">{stats.on_hold}</p>
                        <p className="text-xs text-gray-500 mt-1">משימות מושהות</p>
                    </div>
                    <div className="w-12 h-12 rounded-full bg-purple-500 flex items-center justify-center">
                        <Pause className="w-6 h-6 text-white" />
                    </div>
                </div>
            </div>

            {/* איחור */}
            <div className="bg-white rounded-lg shadow-sm border p-6 hover:shadow-md transition-shadow">
                <div className="flex items-center justify-between">
                    <div className="flex-1">
                        <p className="text-sm font-medium text-gray-600 mb-1">באיחור</p>
                        <p className="text-3xl font-bold text-red-600">{stats.overdue}</p>
                        <p className="text-xs text-gray-500 mt-1">משימות שפגו</p>
                    </div>
                    <div className="w-12 h-12 rounded-full bg-red-500 flex items-center justify-center">
                        <XCircle className="w-6 h-6 text-white" />
                    </div>
                </div>
            </div>

            {/* השבוע */}
            <div className="bg-white rounded-lg shadow-sm border p-6 hover:shadow-md transition-shadow">
                <div className="flex items-center justify-between">
                    <div className="flex-1">
                        <p className="text-sm font-medium text-gray-600 mb-1">השבוע</p>
                        <p className="text-3xl font-bold text-indigo-600">{stats.due_this_week}</p>
                        <p className="text-xs text-gray-500 mt-1">משימות השבוע</p>
                    </div>
                    <div className="w-12 h-12 rounded-full bg-indigo-500 flex items-center justify-center">
                        <Calendar className="w-6 h-6 text-white" />
                    </div>
                </div>
            </div>
        </div>
    );
}
