'use client';

import React, { useEffect, useState } from 'react';
import { RichTextEditor } from '@/components/ui/rich-text-editor';
import { Mic, Camera, Send, MessageSquare, Play, Trash2, Clock } from 'lucide-react';
// Remove direct database imports - we'll use API routes instead

type Mode = 'create' | 'edit';

type TaskStatus = 'pending' | 'in_progress' | 'completed' | 'cancelled' | 'on_hold';
type TaskPriority = 'low' | 'medium' | 'high' | 'urgent';

interface Customer {
    id: string;
    name: string;
}

interface Domain {
    id: string;
    domain: string;
}

interface Project {
    id: string;
    title: string;
}

interface Props {
    mode: Mode;
    taskId?: string;
    defaultCustomerId?: string;
    defaultDomainId?: string;
    defaultProjectId?: string;
    lockCustomer?: boolean;
    lockDomain?: boolean;
    lockProject?: boolean;
}

export default function TaskForm({
    mode,
    taskId,
    defaultCustomerId,
    defaultDomainId,
    defaultProjectId,
    lockCustomer = false,
    lockDomain = false,
    lockProject = false
}: Props) {
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState<string | null>(null);

    // State for comments
    const [comment, setComment] = useState('');
    const [isRecording, setIsRecording] = useState(false);
    const [recordings, setRecordings] = useState<Array<{ id: string, name: string, duration: string, timestamp: Date, audioBlob?: Blob }>>([]);
    const [mediaRecorder, setMediaRecorder] = useState<MediaRecorder | null>(null);
    const [audioChunks, setAudioChunks] = useState<Blob[]>([]);

    // Load recordings when taskId changes
    useEffect(() => {
        if (taskId && mode === 'edit') {
            fetch(`/api/tasks/${taskId}/recordings`)
                .then(response => {
                    console.log('Response status:', response.status);
                    if (!response.ok) {
                        throw new Error(`HTTP ${response.status}`);
                    }
                    return response.json();
                })
                .then(recordings => {
                    console.log('הקלטות נטענו:', recordings);
                    console.log('סוג recordings:', typeof recordings);
                    console.log('האם זה מערך:', Array.isArray(recordings));

                    // וידוא שrecordings הוא מערך
                    if (!Array.isArray(recordings)) {
                        console.warn('Recordings is not an array:', recordings);
                        return;
                    }

                    const formattedRecordings = recordings.map((rec: any) => {
                        console.log('הקלטה:', rec);
                        console.log('סוג audio_data:', typeof rec.audio_data);
                        console.log('גודל audio_data:', rec.audio_data?.length);

                        // המרת Buffer ל-Uint8Array ואז ל-Blob
                        let audioBlob;
                        if (rec.audio_data && rec.audio_data.length > 0) {
                            // אם זה Buffer, המר ל-Uint8Array
                            if (Buffer.isBuffer(rec.audio_data)) {
                                const audioData = new Uint8Array(rec.audio_data);
                                audioBlob = new Blob([audioData], { type: 'audio/webm' });
                            } else if (Array.isArray(rec.audio_data)) {
                                // אם זה מערך של מספרים
                                const audioData = new Uint8Array(rec.audio_data);
                                audioBlob = new Blob([audioData], { type: 'audio/webm' });
                            } else {
                                console.error('פורמט לא ידוע של audio_data:', rec.audio_data);
                                audioBlob = new Blob([], { type: 'audio/webm' });
                            }
                        } else {
                            console.error('audio_data ריק או לא קיים');
                            audioBlob = new Blob([], { type: 'audio/webm' });
                        }

                        return {
                            id: rec.id.toString(),
                            name: rec.name,
                            duration: rec.duration,
                            timestamp: new Date(rec.created_at),
                            audioBlob: audioBlob
                        };
                    });
                    setRecordings(formattedRecordings);
                })
                .catch(error => {
                    console.error('Error loading recordings:', error);
                    console.log('Setting empty recordings array due to error');
                    setRecordings([]);
                });
        }
    }, [taskId, mode]);

    // Comment handlers
    const handleSendComment = () => {
        if (comment.trim()) {
            console.log('שליחת תגובה:', comment);
            setComment('');
            // כאן נוסיף לוגיקה לשמירת התגובה
        }
    };

    const handleVoiceRecording = async () => {
        if (isRecording) {
            console.log('עצירת הקלטה');
            setIsRecording(false);

            if (mediaRecorder && mediaRecorder.state === 'recording') {
                mediaRecorder.stop();
            }
        } else {
            try {
                console.log('התחלת הקלטה');
                const stream = await navigator.mediaDevices.getUserMedia({ audio: true });

                // בדיקת פורמטים נתמכים
                const supportedTypes = ['audio/webm', 'audio/mp4', 'audio/wav'];
                let mimeType = 'audio/webm';

                for (const type of supportedTypes) {
                    if (MediaRecorder.isTypeSupported(type)) {
                        mimeType = type;
                        console.log('פורמט נתמך:', type);
                        break;
                    }
                }

                const recorder = new MediaRecorder(stream, { mimeType });
                const chunks: Blob[] = [];

                recorder.ondataavailable = (event) => {
                    if (event.data.size > 0) {
                        chunks.push(event.data);
                    }
                };

                recorder.onstop = async () => {
                    // שימוש בפורמט הנתמך
                    const audioBlob = new Blob(chunks, { type: mimeType });

                    // הוספת הקלטה חדשה לרשימה
                    const newRecording = {
                        id: Date.now().toString(),
                        name: `הקלטה ${recordings.length + 1}`,
                        duration: '00:15', // משך דמה
                        timestamp: new Date(),
                        audioBlob: audioBlob
                    };
                    setRecordings(prev => [...prev, newRecording]);

                    // שמירה בבסיס הנתונים
                    if (taskId && mode === 'edit') {
                        try {
                            const formData = new FormData();
                            formData.append('name', newRecording.name);
                            formData.append('duration', newRecording.duration);
                            formData.append('audio', audioBlob, 'recording.wav');

                            const response = await fetch(`/api/tasks/${taskId}/recordings`, {
                                method: 'POST',
                                body: formData
                            });

                            console.log('📡 Response status:', response.status);
                            console.log('📡 Response URL:', response.url);

                            if (response.ok) {
                                const data = await response.json();
                                console.log('✅ הקלטה נשמרה בבסיס הנתונים:', data);
                            } else {
                                const errorText = await response.text();
                                console.error('❌ שגיאה בשמירת הקלטה:', response.status, errorText);
                            }
                        } catch (error) {
                            console.error('שגיאה בשמירת הקלטה:', error);
                        }
                    }

                    // עצירת הסטרים
                    stream.getTracks().forEach(track => track.stop());
                };

                setMediaRecorder(recorder);
                setAudioChunks(chunks);
                recorder.start();
                setIsRecording(true);
            } catch (error) {
                console.error('שגיאה בהקלטה:', error);
                alert('לא ניתן לגשת למיקרופון. אנא בדוק את ההרשאות.');
            }
        }
    };

    const handleFileUpload = () => {
        console.log('העלאת קובץ');
        // כאן נוסיף לוגיקה להעלאת קבצים
    };

    const handleDeleteRecording = async (id: string) => {
        try {
            // מחיקה מבסיס הנתונים
            const response = await fetch(`/api/tasks/recordings/${id}`, {
                method: 'DELETE'
            });

            if (response.ok) {
                console.log('הקלטה נמחקה מבסיס הנתונים');
                // מחיקה מהרשימה המקומית
                setRecordings(prev => prev.filter(rec => rec.id !== id));
            } else {
                console.error('שגיאה במחיקת הקלטה');
            }
        } catch (error) {
            console.error('שגיאה במחיקת הקלטה:', error);
        }
    };

    const handlePlayRecording = (id: string) => {
        console.log('השמעת הקלטה:', id);
        const recording = recordings.find(rec => rec.id === id);
        if (recording && recording.audioBlob) {
            try {
                console.log('גודל הקלטה:', recording.audioBlob.size, 'bytes');
                console.log('סוג הקלטה:', recording.audioBlob.type);

                // בדיקה שהבלוב לא ריק
                if (recording.audioBlob.size === 0) {
                    alert('הקלטה ריקה - לא ניתן להשמיע');
                    return;
                }

                // יצירת URL מהבלוב
                const audioUrl = URL.createObjectURL(recording.audioBlob);
                console.log('URL נוצר:', audioUrl);

                const audio = new Audio(audioUrl);

                // הוספת event listeners
                audio.onloadeddata = () => {
                    console.log('הקלטה נטענה בהצלחה');
                };

                audio.oncanplay = () => {
                    console.log('הקלטה מוכנה להשמעה');
                };

                audio.onerror = (error) => {
                    console.error('שגיאה בטעינת הקלטה:', error);
                    console.error('פרטי השגיאה:', {
                        error: error,
                        code: audio.error?.code,
                        message: audio.error?.message
                    });
                    alert('שגיאה בטעינת ההקלטה. ייתכן שהפורמט לא נתמך.');
                };

                // ניסיון השמעה
                audio.play().then(() => {
                    console.log('הקלטה מתחילה להשמע');
                }).catch(error => {
                    console.error('שגיאה בהשמעה:', error);
                    alert('לא ניתן להשמיע את ההקלטה. בדוק שהדפדפן תומך בפורמט האודיו.');
                });
            } catch (error) {
                console.error('שגיאה ביצירת URL:', error);
                alert('שגיאה ביצירת קובץ השמעה');
            }
        } else {
            alert('הקלטה לא נמצאה');
        }
    };

    const [title, setTitle] = useState('');
    const [description, setDescription] = useState('');
    const [customerId, setCustomerId] = useState<string>(defaultCustomerId || '');
    const [domainId, setDomainId] = useState<string>(defaultDomainId || '');
    const [projectId, setProjectId] = useState<string>(defaultProjectId || '');
    const [status, setStatus] = useState<TaskStatus>('pending');
    const [priority, setPriority] = useState<TaskPriority>('medium');
    const [dueDate, setDueDate] = useState<string>('');
    const [assignedTo, setAssignedTo] = useState('');
    const [progress, setProgress] = useState<number>(0);
    const [category, setCategory] = useState('');

    const [customers, setCustomers] = useState<Customer[]>([]);
    const [domains, setDomains] = useState<Domain[]>([]);
    const [projects, setProjects] = useState<Project[]>([]);

    // קבוצות קטגוריות
    const categoryGroups = [
        {
            label: '👥 ניהול לקוחות ומכירות',
            categories: [
                { value: 'customer', label: 'לקוחות', icon: '👥' },
                { value: 'sales', label: 'מכירות', icon: '💼' },
                { value: 'finance', label: 'כספים', icon: '💰' },
            ]
        },
        {
            label: '🌐 דומיינים ואימייל',
            categories: [
                { value: 'domain', label: 'דומיינים', icon: '🌐' },
                { value: 'email', label: 'אימייל', icon: '📧' },
            ]
        },
        {
            label: '💻 טכנולוגיה ופיתוח',
            categories: [
                { value: 'technical', label: 'טכני', icon: '💻' },
                { value: 'development', label: 'פיתוח', icon: '🎨' },
                { value: 'maintenance', label: 'תחזוקה', icon: '🔧' },
            ]
        },
        {
            label: '🔒 אבטחה ואיכות',
            categories: [
                { value: 'security', label: 'אבטחה', icon: '🔒' },
                { value: 'quality', label: 'בקרת איכות', icon: '✅' },
            ]
        },
        {
            label: '📊 ניהול ותפעול',
            categories: [
                { value: 'project', label: 'פרויקטים', icon: '📊' },
                { value: 'operations', label: 'תפעול', icon: '⚙️' },
                { value: 'administrative', label: 'מנהלתי', icon: '📝' },
            ]
        },
        {
            label: '📢 שיווק ותמיכה',
            categories: [
                { value: 'marketing', label: 'שיווק', icon: '📢' },
                { value: 'support', label: 'תמיכה', icon: '🎓' },
            ]
        },
        {
            label: '🎓 מחקר והדרכה',
            categories: [
                { value: 'research', label: 'מחקר', icon: '🔬' },
                { value: 'training', label: 'הדרכה פנימית', icon: '🎓' },
                { value: 'legal', label: 'משפטי', icon: '⚖️' },
            ]
        },
        {
            label: '📌 אחר',
            categories: [
                { value: 'other', label: 'אחר', icon: '📌' },
            ]
        }
    ];

    // פונקציה עזר לקבלת שם קטגוריה עם אייקון
    const getCategoryLabel = (categoryValue: string) => {
        for (const group of categoryGroups) {
            const category = group.categories.find(cat => cat.value === categoryValue);
            if (category) {
                return `${category.icon} ${category.label}`;
            }
        }
        return categoryValue;
    };

    // טעינת לקוחות בטעינה ראשונית
    useEffect(() => {
        (async () => {
            try {
                const res = await fetch('/api/comprehensive/customers');
                if (res.ok) {
                    const data = await res.json();
                    setCustomers(data);
                }
            } catch (e) {
                console.error('Error loading customers:', e);
            }
        })();
    }, []);

    // טעינת דומיינים ופרויקטים כשבוחרים לקוח
    useEffect(() => {
        if (customerId) {
            (async () => {
                try {
                    const [domainsRes, projectsRes] = await Promise.all([
                        fetch(`/api/customers/${customerId}/domains`),
                        fetch(`/api/customers/${customerId}/projects`)
                    ]);
                    if (domainsRes.ok) {
                        const domainsData = await domainsRes.json();
                        setDomains(domainsData);
                    }
                    if (projectsRes.ok) {
                        const projectsData = await projectsRes.json();
                        setProjects(projectsData);
                    }
                } catch (e) {
                    console.error('Error loading customer data:', e);
                }
            })();
        } else {
            setDomains([]);
            setProjects([]);
            setDomainId('');
            setProjectId('');
        }
    }, [customerId]);

    // טעינת משימה קיימת במצב עריכה
    useEffect(() => {
        if (mode === 'edit' && taskId) {
            (async () => {
                setLoading(true);
                try {
                    const res = await fetch(`/api/tasks/${taskId}`, { cache: 'no-store' });
                    if (!res.ok) throw new Error('failed');
                    const t = await res.json();
                    setTitle(t.title ?? '');
                    setDescription(t.description ?? '');
                    setCustomerId(t.customer_id ?? '');
                    setDomainId(t.domain_id ?? '');
                    setProjectId(t.project_id ?? '');
                    setStatus((t.status ?? 'pending') as TaskStatus);
                    setPriority((t.priority ?? 'medium') as TaskPriority);
                    setDueDate(t.due_date ? String(t.due_date).slice(0, 10) : '');
                    setAssignedTo(t.assigned_to ?? '');
                    setProgress(Number(t.progress ?? 0));
                    setCategory(t.category ?? '');
                } catch (e) {
                    setError('משימה לא נמצאה. ייתכן שהיא נמחקה או שהמזהה שגוי.');
                } finally {
                    setLoading(false);
                }
            })();
        }
    }, [mode, taskId]);

    const onSubmit = async (e: React.FormEvent) => {
        e.preventDefault();

        // וולידציה: customer_id חובה
        if (!customerId) {
            setError('חובה לבחור לקוח');
            return;
        }

        setLoading(true);
        setError(null);
        try {
            const payload = {
                title,
                description,
                status,
                priority,
                category_id: category || null,
                customer_id: customerId,
                project_id: projectId || null,
                assigned_to: assignedTo || null,
                due_date: dueDate || null,
                completed_at: status === 'completed' ? new Date().toISOString() : null
            };
            const res = await fetch(mode === 'create' ? '/api/tasks' : `/api/tasks/${taskId}`, {
                method: mode === 'create' ? 'POST' : 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(payload),
            });
            if (!res.ok) {
                const errData = await res.json();
                throw new Error(errData.error || 'שמירה נכשלה');
            }
            window.location.href = '/tasks';
        } catch (e: any) {
            setError(e.message || 'שמירה נכשלה');
        } finally {
            setLoading(false);
        }
    };

    return (
        <form onSubmit={onSubmit} className="bg-white shadow rounded-lg p-6 space-y-6" dir="rtl">
            {error && (
                <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded">
                    <div className="flex items-center justify-between">
                        <span>{error}</span>
                        {error.includes('לא נמצאה') && (
                            <a
                                href="/tasks"
                                className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 transition-colors"
                            >
                                חזור לרשימת המשימות
                            </a>
                        )}
                    </div>
                </div>
            )}

            {/* כותרת ותיאור */}
            <div className="space-y-4">
                <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">כותרת המשימה *</label>
                    <input
                        className="w-full border border-gray-300 rounded-md px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        value={title}
                        onChange={e => setTitle(e.target.value)}
                        required
                        placeholder="הזן כותרת למשימה"
                    />
                </div>
                <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">תיאור</label>
                    <RichTextEditor
                        value={description}
                        onChange={setDescription}
                        placeholder="תאר את המשימה..."
                        className="w-full"
                    />
                </div>
            </div>

            {/* קשרים: לקוח, דומיין, פרויקט */}
            <div className="border-t pt-4">
                <h3 className="text-lg font-medium text-gray-900 mb-4">קשרים ללקוח ופרויקטים</h3>
                <div className="space-y-4">
                    {/* בחירת לקוח */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">
                            לקוח * {lockCustomer && <span className="text-xs text-gray-500">(נעול)</span>}
                        </label>
                        <select
                            className="w-full border border-gray-300 rounded-md px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent disabled:bg-gray-100 disabled:cursor-not-allowed"
                            value={customerId}
                            onChange={e => setCustomerId(e.target.value)}
                            required
                            disabled={lockCustomer}
                            aria-label="בחירת לקוח"
                        >
                            <option value="">-- בחר לקוח --</option>
                            {customers.map(c => (
                                <option key={c.id} value={c.id}>{c.name}</option>
                            ))}
                        </select>
                    </div>

                    {/* בחירת דומיין (אופציונלי) */}
                    {customerId && (
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-1">
                                דומיין (אופציונלי) {lockDomain && <span className="text-xs text-gray-500">(נעול)</span>}
                            </label>
                            <select
                                className="w-full border border-gray-300 rounded-md px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent disabled:bg-gray-100 disabled:cursor-not-allowed"
                                value={domainId}
                                onChange={e => setDomainId(e.target.value)}
                                disabled={lockDomain}
                                aria-label="בחירת דומיין"
                            >
                                <option value="">-- ללא דומיין --</option>
                                {domains.map(d => (
                                    <option key={d.id} value={d.id}>{d.domain}</option>
                                ))}
                            </select>
                            {domains.length === 0 && (
                                <p className="text-xs text-gray-500 mt-1">אין דומיינים ללקוח זה</p>
                            )}
                        </div>
                    )}

                    {/* בחירת פרויקט (אופציונלי) */}
                    {customerId && (
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-1">
                                פרויקט (אופציונלי) {lockProject && <span className="text-xs text-gray-500">(נעול)</span>}
                            </label>
                            <select
                                className="w-full border border-gray-300 rounded-md px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent disabled:bg-gray-100 disabled:cursor-not-allowed"
                                value={projectId}
                                onChange={e => setProjectId(e.target.value)}
                                disabled={lockProject}
                                aria-label="בחירת פרויקט"
                            >
                                <option value="">-- ללא פרויקט --</option>
                                {projects.map(p => (
                                    <option key={p.id} value={p.id}>{p.title}</option>
                                ))}
                            </select>
                            {projects.length === 0 && (
                                <p className="text-xs text-gray-500 mt-1">אין פרויקטים ללקוח זה</p>
                            )}
                        </div>
                    )}
                </div>
            </div>

            {/* פרטי משימה */}
            <div className="border-t pt-4">
                <h3 className="text-lg font-medium text-gray-900 mb-4">פרטי המשימה</h3>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">סטטוס</label>
                        <select className="w-full border border-gray-300 rounded-md px-3 py-2" value={status} onChange={e => setStatus(e.target.value as TaskStatus)} aria-label="בחירת סטטוס">
                            <option value="pending">ממתין</option>
                            <option value="in_progress">בתהליך</option>
                            <option value="completed">הושלם</option>
                            <option value="on_hold">בהמתנה</option>
                            <option value="cancelled">בוטל</option>
                        </select>
                    </div>
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">עדיפות</label>
                        <select className="w-full border border-gray-300 rounded-md px-3 py-2" value={priority} onChange={e => setPriority(e.target.value as TaskPriority)} aria-label="בחירת עדיפות">
                            <option value="urgent">דחוף</option>
                            <option value="high">גבוה</option>
                            <option value="medium">בינוני</option>
                            <option value="low">נמוך</option>
                        </select>
                    </div>
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">קטגוריה</label>
                        <select
                            className="w-full border border-gray-300 rounded-md px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            value={category}
                            onChange={e => setCategory(e.target.value)}
                            aria-label="בחירת קטגוריה"
                        >
                            <option value="">-- בחר קטגוריה --</option>
                            {categoryGroups.map(group => (
                                <optgroup key={group.label} label={group.label}>
                                    {group.categories.map(cat => (
                                        <option key={cat.value} value={cat.value}>
                                            {cat.icon} {cat.label}
                                        </option>
                                    ))}
                                </optgroup>
                            ))}
                        </select>
                        {category && (
                            <div className="mt-2 text-sm text-gray-600">
                                נבחר: {getCategoryLabel(category)}
                            </div>
                        )}
                    </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mt-4">
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">תאריך יעד</label>
                        <input
                            type="date"
                            className="w-full border border-gray-300 rounded-md px-3 py-2"
                            value={dueDate}
                            onChange={e => setDueDate(e.target.value)}
                            aria-label="תאריך יעד"
                        />
                    </div>
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">אחראי</label>
                        <input
                            className="w-full border border-gray-300 rounded-md px-3 py-2"
                            value={assignedTo}
                            onChange={e => setAssignedTo(e.target.value)}
                            placeholder="שם האחראי"
                            aria-label="שם האחראי"
                        />
                    </div>
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">התקדמות (%)</label>
                        <input
                            type="number"
                            min={0}
                            max={100}
                            className="w-full border border-gray-300 rounded-md px-3 py-2"
                            value={progress}
                            onChange={e => setProgress(Number(e.target.value))}
                            aria-label="התקדמות באחוזים"
                        />
                    </div>
                </div>
            </div>

            {/* תגובות */}
            <div className="border-t pt-6 mt-6">
                <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
                    <MessageSquare className="w-5 h-5" />
                    תגובות
                </h3>

                <div className="bg-gray-50 rounded-lg p-4">
                    <div className="flex items-center gap-3">
                        <input
                            type="text"
                            value={comment}
                            onChange={(e) => setComment(e.target.value)}
                            placeholder="כתוב תגובה..."
                            className="flex-1 border border-gray-300 rounded-md px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            onKeyPress={(e) => {
                                if (e.key === 'Enter') {
                                    handleSendComment();
                                }
                            }}
                        />
                        <button
                            type="button"
                            onClick={handleSendComment}
                            className="bg-blue-600 text-white p-2 rounded-md hover:bg-blue-700 transition-colors"
                            title="שלח תגובה"
                        >
                            <Send className="w-4 h-4" />
                        </button>
                        <button
                            type="button"
                            onClick={handleVoiceRecording}
                            className={`p-2 rounded-md transition-colors ${isRecording
                                ? 'bg-red-600 text-white hover:bg-red-700'
                                : 'bg-gray-200 text-gray-700 hover:bg-gray-300'
                                }`}
                            title={isRecording ? "עצור הקלטה" : "התחל הקלטה"}
                        >
                            <Mic className="w-4 h-4" />
                        </button>
                        <button
                            type="button"
                            onClick={handleFileUpload}
                            className="bg-gray-200 text-gray-700 p-2 rounded-md hover:bg-gray-300 transition-colors"
                            title="העלה קובץ"
                        >
                            <Camera className="w-4 h-4" />
                        </button>
                    </div>
                </div>

                {/* רשימת הקלטות */}
                {recordings.length > 0 && (
                    <div className="mt-4">
                        <h4 className="text-sm font-medium text-gray-700 mb-2">הקלטות קוליות</h4>
                        <div className="space-y-2">
                            {recordings.map((recording) => (
                                <div key={recording.id} className="flex items-center justify-between bg-white border border-gray-200 rounded-lg p-3">
                                    <div className="flex items-center gap-3">
                                        <button
                                            type="button"
                                            onClick={() => handlePlayRecording(recording.id)}
                                            className="bg-blue-100 text-blue-600 p-2 rounded-full hover:bg-blue-200 transition-colors"
                                            title="השמע הקלטה"
                                        >
                                            <Play className="w-4 h-4" />
                                        </button>
                                        <div>
                                            <div className="font-medium text-gray-900">{recording.name}</div>
                                            <div className="flex items-center gap-2 text-sm text-gray-500">
                                                <Clock className="w-3 h-3" />
                                                <span>{recording.duration}</span>
                                                <span>•</span>
                                                <span>{recording.timestamp.toLocaleString('he-IL')}</span>
                                            </div>
                                        </div>
                                    </div>
                                    <button
                                        type="button"
                                        onClick={() => handleDeleteRecording(recording.id)}
                                        className="text-red-500 hover:text-red-700 p-1 rounded transition-colors"
                                        title="מחק הקלטה"
                                    >
                                        <Trash2 className="w-4 h-4" />
                                    </button>
                                </div>
                            ))}
                        </div>
                    </div>
                )}
            </div>

            {/* כפתורים */}
            <div className="flex justify-start gap-3 border-t pt-4">
                <button
                    type="submit"
                    disabled={loading}
                    className="bg-blue-600 text-white px-6 py-2 rounded-md hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                >
                    {loading ? 'שומר...' : (mode === 'create' ? 'צור משימה' : 'שמור שינויים')}
                </button>
                <a
                    href="/tasks"
                    className="px-6 py-2 border border-gray-300 rounded-md hover:bg-gray-50 transition-colors"
                >
                    ביטול
                </a>
            </div>
        </form>
    );
}


