'use client';

import React, { useState } from 'react';
import { X, Star, Clock, Mic, Camera, Send, User, Calendar, Flag, Tag, Users, Timer, Bell, MessageSquare } from 'lucide-react';

interface TaskDetailsModalProps {
    isOpen: boolean;
    onClose: () => void;
    task: {
        id: number;
        title: string;
        description?: string;
        status: string;
        priority: string;
        due_date?: string;
        assigned_to?: string;
        progress: number;
        category?: string;
        customer_name?: string;
        domain?: string;
        project_title?: string;
    } | null;
}

export default function TaskDetailsModal({ isOpen, onClose, task }: TaskDetailsModalProps) {
    const [comment, setComment] = useState('');
    const [isTimerRunning, setIsTimerRunning] = useState(false);
    const [loggedTime, setLoggedTime] = useState('00:00:09');

    if (!isOpen || !task) return null;

    const getStatusLabel = (status: string) => {
        switch (status) {
            case 'pending': return 'ממתין';
            case 'in_progress': return 'בתהליך';
            case 'completed': return 'הושלם';
            case 'on_hold': return 'בהמתנה';
            case 'cancelled': return 'בוטל';
            default: return status;
        }
    };

    const getPriorityLabel = (priority: string) => {
        switch (priority) {
            case 'urgent': return 'דחוף';
            case 'high': return 'גבוה';
            case 'medium': return 'בינוני';
            case 'low': return 'נמוך';
            default: return priority;
        }
    };

    const handleAddComment = () => {
        if (comment.trim()) {
            // כאן תהיה לוגיקה לשמירת התגובה
            console.log('Adding comment:', comment);
            setComment('');
            // הוספת הודעת הצלחה
            alert('התגובה נוספה בהצלחה!');
        }
    };

    const handleStartTimer = () => {
        setIsTimerRunning(!isTimerRunning);
        // כאן תהיה לוגיקה לניהול הטיימר
        if (!isTimerRunning) {
            alert('הטיימר התחיל!');
        } else {
            alert('הטיימר נעצר!');
        }
    };

    const handleVoiceRecord = () => {
        alert('פונקציונליות הקלטה קולית - בפיתוח');
    };

    const handleFileUpload = () => {
        alert('פונקציונליות העלאת קובץ - בפיתוח');
    };

    const handleEditTask = () => {
        // פתיחת דף עריכה בתוך החלון המודאלי
        window.open(`/tasks/${task.id}/edit`, '_blank');
    };

    const handleDuplicateTask = () => {
        alert('פונקציונליות שכפול משימה - בפיתוח');
    };

    return (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50" dir="rtl">
            <div className="bg-white rounded-lg shadow-xl w-full max-w-4xl max-h-[90vh] overflow-hidden">
                {/* Header */}
                <div className="flex items-center justify-between p-6 border-b">
                    <div className="flex items-center gap-3">
                        <h2 className="text-xl font-bold">מידע על המשימה #{task.id}</h2>
                        <Star className="w-5 h-5 text-yellow-500" />
                    </div>
                    <button
                        onClick={onClose}
                        className="text-gray-400 hover:text-gray-600"
                        aria-label="סגור"
                        title="סגור"
                    >
                        <X className="w-6 h-6" />
                    </button>
                </div>

                {/* Task Title */}
                <div className="p-6 border-b">
                    <h3 className="text-lg font-semibold text-gray-900">{task.title}</h3>
                </div>

                {/* Content */}
                <div className="p-6 space-y-6 max-h-[60vh] overflow-y-auto">
                    {/* Responsible */}
                    <div className="flex items-center gap-3">
                        <User className="w-5 h-5 text-gray-400" />
                        <span className="text-sm font-medium text-gray-700">הוסף אחראי:</span>
                        <div className="flex items-center gap-2">
                            <div className="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center">
                                <User className="w-4 h-4 text-blue-600" />
                            </div>
                            <span className="text-sm">{task.assigned_to || 'טיטי'}</span>
                            <button className="px-3 py-1 bg-blue-100 text-blue-800 rounded-full text-xs">
                                {getStatusLabel(task.status)}
                            </button>
                        </div>
                    </div>

                    {/* Project */}
                    <div className="flex items-center gap-3">
                        <span className="text-sm font-medium text-gray-700 w-24">פרויקט:</span>
                        <span className="text-sm text-gray-900">{task.project_title || 'בנית אתר להלגה'}</span>
                    </div>

                    {/* Milestone */}
                    <div className="flex items-center gap-3">
                        <span className="text-sm font-medium text-gray-700 w-24">אבן דרך:</span>
                        <button className="text-sm text-blue-600 hover:text-blue-800">
                            הוסף אבן דרך
                        </button>
                    </div>

                    {/* Start Date */}
                    <div className="flex items-center gap-3">
                        <Calendar className="w-5 h-5 text-gray-400" />
                        <span className="text-sm font-medium text-gray-700 w-24">תאריך התחלה:</span>
                        <button className="text-sm text-blue-600 hover:text-blue-800">
                            הוסף תאריך התחלה
                        </button>
                    </div>

                    {/* Due Date */}
                    <div className="flex items-center gap-3">
                        <Calendar className="w-5 h-5 text-gray-400" />
                        <span className="text-sm font-medium text-gray-700 w-24">מועד אחרון:</span>
                        <span className="text-sm text-gray-900">
                            {task.due_date ? new Date(task.due_date).toLocaleDateString('he-IL') : 'הוסף מועד אחרון'}
                        </span>
                    </div>

                    {/* Priority */}
                    <div className="flex items-center gap-3">
                        <Flag className="w-5 h-5 text-gray-400" />
                        <span className="text-sm font-medium text-gray-700 w-24">עדיפות:</span>
                        <select className="text-sm border rounded px-2 py-1" aria-label="עדיפות">
                            <option value="low">נמוך</option>
                            <option value="medium">בינוני</option>
                            <option value="high">גבוה</option>
                            <option value="urgent">דחוף</option>
                        </select>
                    </div>

                    {/* Label */}
                    <div className="flex items-center gap-3">
                        <Tag className="w-5 h-5 text-gray-400" />
                        <span className="text-sm font-medium text-gray-700 w-24">תווית:</span>
                        <button className="text-sm text-blue-600 hover:text-blue-800">
                            הוסף תווית
                        </button>
                    </div>

                    {/* Collaborators */}
                    <div className="flex items-center gap-3">
                        <Users className="w-5 h-5 text-gray-400" />
                        <span className="text-sm font-medium text-gray-700 w-24">משתפי פעולה:</span>
                        <button className="text-sm text-blue-600 hover:text-blue-800">
                            הוסף משתפי פעולה
                        </button>
                    </div>

                    {/* Timer Section */}
                    <div className="flex items-center gap-3">
                        <Timer className="w-5 h-5 text-gray-400" />
                        <button
                            onClick={handleStartTimer}
                            className={`px-4 py-2 rounded-full text-sm font-medium ${isTimerRunning
                                ? 'bg-red-100 text-red-800 hover:bg-red-200'
                                : 'bg-blue-100 text-blue-800 hover:bg-blue-200'
                                }`}
                        >
                            {isTimerRunning ? 'עצור טיימר' : 'התחל טיימר'}
                        </button>
                        <span className="text-sm text-gray-700">סך כל הזמן שנרשם: {loggedTime}</span>
                    </div>

                    {/* Reminders */}
                    <div className="flex items-center gap-3">
                        <Bell className="w-5 h-5 text-gray-400" />
                        <span className="text-sm font-medium text-gray-700">תזכורות (פרטי):</span>
                        <button className="text-sm text-blue-600 hover:text-blue-800">
                            הוסף תזכורת
                        </button>
                    </div>

                    {/* Comment Section */}
                    <div className="space-y-3">
                        <div className="flex items-center gap-2">
                            <MessageSquare className="w-5 h-5 text-gray-400" />
                            <span className="text-sm font-medium text-gray-700">כתוב תגובה..</span>
                        </div>
                        <div className="flex gap-2">
                            <input
                                type="text"
                                value={comment}
                                onChange={(e) => setComment(e.target.value)}
                                placeholder="כתוב תגובה.."
                                className="flex-1 border rounded px-3 py-2 text-sm"
                                dir="rtl"
                            />
                            <button
                                onClick={handleAddComment}
                                className="px-3 py-2 bg-blue-600 text-white rounded hover:bg-blue-700"
                                aria-label="הוסף תגובה"
                                title="הוסף תגובה"
                            >
                                <Send className="w-4 h-4" />
                            </button>
                            <button
                                onClick={handleVoiceRecord}
                                className="px-3 py-2 border rounded hover:bg-gray-50"
                                aria-label="הקלטה קולית"
                                title="הקלטה קולית"
                            >
                                <Mic className="w-4 h-4" />
                            </button>
                            <button
                                onClick={handleFileUpload}
                                className="px-3 py-2 border rounded hover:bg-gray-50"
                                aria-label="העלאת קובץ"
                                title="העלאת קובץ"
                            >
                                <Camera className="w-4 h-4" />
                            </button>
                        </div>
                    </div>

                    {/* Activity Section */}
                    <div className="space-y-3">
                        <h4 className="text-sm font-medium text-gray-700">פעילות</h4>
                        <div className="bg-gray-50 rounded p-3">
                            <div className="text-xs text-gray-500">titi steinfeld 2024-11-15 01:15:43 pm</div>
                            <div className="text-sm text-gray-900">עודכן משימה: #{task.id} - טיטי</div>
                            <div className="text-sm text-gray-700">עדיפות {getPriorityLabel(task.priority)}</div>
                        </div>
                    </div>
                </div>

                {/* Footer */}
                <div className="flex items-center justify-between p-6 border-t bg-gray-50">
                    <div className="flex gap-3">
                        <button
                            onClick={onClose}
                            className="px-4 py-2 text-gray-600 hover:text-gray-800"
                        >
                            <X className="w-4 h-4 inline ml-1" />
                            סגור
                        </button>
                        <button
                            onClick={handleEditTask}
                            className="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700"
                        >
                            ערוך משימה
                        </button>
                        <button
                            onClick={handleDuplicateTask}
                            className="px-4 py-2 border rounded hover:bg-gray-50"
                        >
                            שכפל משימה
                        </button>
                    </div>
                </div>
            </div>
        </div>
    );
}
