'use client';

import { useState, useEffect } from 'react';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { X, Plus } from 'lucide-react';

interface ProjectTag {
    id: number;
    name: string;
    color: string;
    description?: string;
}

interface TagManagerProps {
    selectedTags: string[];
    onChange: (tags: string[]) => void;
    placeholder?: string;
}

export function TagManager({ selectedTags, onChange, placeholder = "הקלד תגית ולחץ Enter" }: TagManagerProps) {
    const [availableTags, setAvailableTags] = useState<ProjectTag[]>([]);
    const [newTag, setNewTag] = useState('');
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        const loadTags = async () => {
            try {
                const response = await fetch('/api/project-tags');
                if (response.ok) {
                    const tags = await response.json();
                    setAvailableTags(tags);
                }
            } catch (error) {
                console.error('Error loading tags:', error);
            } finally {
                setLoading(false);
            }
        };

        loadTags();
    }, []);

    const handleAddTag = (tagName: string) => {
        if (tagName.trim() && !selectedTags.includes(tagName.trim())) {
            onChange([...selectedTags, tagName.trim()]);
        }
    };

    const handleRemoveTag = (tagToRemove: string) => {
        onChange(selectedTags.filter(tag => tag !== tagToRemove));
    };

    const handleKeyPress = (e: React.KeyboardEvent) => {
        if (e.key === 'Enter') {
            e.preventDefault();
            handleAddTag(newTag);
            setNewTag('');
        }
    };

    const handleAddExistingTag = (tag: ProjectTag) => {
        if (!selectedTags.includes(tag.name)) {
            onChange([...selectedTags, tag.name]);
        }
    };

    return (
        <div className="space-y-4">
            {/* תגיות נבחרות */}
            {selectedTags.length > 0 && (
                <div className="flex flex-wrap gap-2">
                    {selectedTags.map((tag) => (
                        <Badge
                            key={tag}
                            variant="secondary"
                            className="flex items-center gap-1"
                        >
                            {tag}
                            <Button
                                type="button"
                                variant="ghost"
                                size="sm"
                                className="h-4 w-4 p-0 hover:bg-transparent"
                                onClick={() => handleRemoveTag(tag)}
                            >
                                <X className="h-3 w-3" />
                            </Button>
                        </Badge>
                    ))}
                </div>
            )}

            {/* הוספת תגית חדשה */}
            <div className="flex gap-2">
                <Input
                    value={newTag}
                    onChange={(e) => setNewTag(e.target.value)}
                    onKeyPress={handleKeyPress}
                    placeholder={placeholder}
                    className="flex-1"
                />
                <Button
                    type="button"
                    variant="outline"
                    size="sm"
                    onClick={() => {
                        handleAddTag(newTag);
                        setNewTag('');
                    }}
                >
                    <Plus className="h-4 w-4" />
                </Button>
            </div>

            {/* תגיות זמינות */}
            {!loading && availableTags.length > 0 && (
                <div className="space-y-2">
                    <label className="text-sm font-medium text-gray-700">תגיות זמינות:</label>
                    <div className="flex flex-wrap gap-2">
                        {availableTags
                            .filter(tag => !selectedTags.includes(tag.name))
                            .map((tag) => (
                                <Badge
                                    key={tag.id}
                                    variant="outline"
                                    className="cursor-pointer hover:bg-gray-100"
                                    style={{ backgroundColor: tag.color + '20', borderColor: tag.color }}
                                    onClick={() => handleAddExistingTag(tag)}
                                >
                                    {tag.name}
                                </Badge>
                            ))}
                    </div>
                </div>
            )}
        </div>
    );
}
