'use client';

import { useState, useEffect } from 'react';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import {
    Dialog,
    DialogContent,
    DialogDescription,
    DialogHeader,
    DialogTitle,
} from '@/components/ui/dialog';
import {
    Form,
    FormControl,
    FormField,
    FormItem,
    FormLabel,
    FormMessage,
} from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Checkbox } from '@/components/ui/checkbox';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Button } from '@/components/ui/button';
import { Textarea } from '@/components/ui/textarea';
import { RichTextEditor } from '@/components/ui/rich-text-editor';
import { CustomerSelect } from '@/components/customers/CustomerSelect';
import { DomainSelect } from '@/components/domains/DomainSelect';

const projectSchema = z.object({
    title: z.string().min(2, 'כותרת חייבת להכיל לפחות 2 תווים'),
    is_open: z.boolean().default(true),
    description: z.string().optional(),
    customer_id: z.string().min(1, 'בחר לקוח'),
    domain_id: z.string().optional(),
    project_type: z.string().min(1, 'בחר סוג פרויקט'),
    start_date: z.string().optional(),
    end_date: z.string().optional(),
    budget: z.number().min(0).optional(),
    tags: z.array(z.string()).optional(),
    status: z.enum(['open', 'in_progress', 'pending', 'completed', 'cancelled']),
    priority: z.enum(['low', 'medium', 'high', 'urgent']).default('medium'),
});

type ProjectForm = z.infer<typeof projectSchema>;

interface Project {
    id: number;
    title: string;
    description?: string;
    customer_id: string;
    domain_id?: string;
    project_type: string;
    status: string;
    start_date?: string;
    end_date?: string;
    budget: number;
    is_open: boolean;
    priority: string;
    tags: string[];
}

interface ProjectDrawerProps {
    project?: Project | null;
    open: boolean;
    onOpenChange: (open: boolean) => void;
    onSuccess?: () => void;
    isViewMode?: boolean;
}

export function ProjectDrawer({ project, open, onOpenChange, onSuccess, isViewMode = false }: ProjectDrawerProps) {
    const [loading, setLoading] = useState(false);

    const form = useForm<ProjectForm>({
        resolver: zodResolver(projectSchema),
        defaultValues: {
            title: '',
            is_open: true,
            description: '',
            customer_id: '',
            domain_id: '',
            project_type: '',
            start_date: '',
            end_date: '',
            budget: 0,
            tags: [],
            status: 'open',
            priority: 'medium',
        },
    });

    // עדכון ערכי ברירת מחדל כשהפרויקט משתנה
    useEffect(() => {
        if (project) {
            form.reset({
                title: project.title,
                is_open: project.is_open,
                description: project.description || '',
                customer_id: project.customer_id,
                domain_id: project.domain_id || '',
                project_type: project.project_type,
                start_date: project.start_date || '',
                end_date: project.end_date || '',
                budget: project.budget || 0,
                tags: project.tags || [],
                status: project.status as any,
                priority: project.priority as any,
            });
        } else {
            form.reset({
                title: '',
                is_open: true,
                description: '',
                customer_id: '',
                domain_id: '',
                project_type: '',
                start_date: '',
                end_date: '',
                budget: 0,
                tags: [],
                status: 'open',
                priority: 'medium',
            });
        }
    }, [project, form]);

    const onSubmit = async (data: ProjectForm) => {
        try {
            setLoading(true);

            const url = project ? `/api/projects/${project.id}` : '/api/projects';
            const method = project ? 'PUT' : 'POST';

            const response = await fetch(url, {
                method,
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    ...data,
                    budget: data.budget || 0,
                }),
            });

            if (response.ok) {
                onOpenChange(false);
                form.reset();
                onSuccess?.();
            } else {
                const error = await response.json();
                console.error('Error saving project:', error);
                alert('שגיאה בשמירת הפרויקט: ' + (error.error || 'שגיאה לא ידועה'));
            }
        } catch (error) {
            console.error('Error saving project:', error);
            alert('שגיאה בשמירת הפרויקט');
        } finally {
            setLoading(false);
        }
    };

    const formatDateForInput = (dateString?: string) => {
        if (!dateString) return '';
        const date = new Date(dateString);
        return date.toISOString().split('T')[0];
    };

    return (
        <Dialog open={open} onOpenChange={onOpenChange}>
            <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto" dir="rtl">
                <DialogHeader>
                    <DialogTitle>
                        {isViewMode ? 'צפה בפרויקט' : (project ? 'ערוך פרויקט' : 'פרויקט חדש')}
                    </DialogTitle>
                    <DialogDescription>
                        {isViewMode ? `צפייה בפרויקט #${project?.id}` : (project ? `עריכת פרויקט #${project.id}` : 'צור פרויקט חדש')}
                    </DialogDescription>
                </DialogHeader>

                <Form {...form}>
                    <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6 mt-6">

                        {/* כותרת */}
                        <FormField
                            control={form.control}
                            name="title"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>כותרת *</FormLabel>
                                    <FormControl>
                                        <Input placeholder="לדוגמה: אתר חדש לחברת ABC" {...field} />
                                    </FormControl>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />

                        {/* פרויקט פתוח */}
                        <FormField
                            control={form.control}
                            name="is_open"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-start space-x-3 space-y-0">
                                    <FormControl>
                                        <Checkbox
                                            checked={field.value}
                                            onCheckedChange={field.onChange}
                                        />
                                    </FormControl>
                                    <div className="space-y-1 leading-none">
                                        <FormLabel>פרויקט פתוח</FormLabel>
                                    </div>
                                </FormItem>
                            )}
                        />

                        {/* תיאור */}
                        <FormField
                            control={form.control}
                            name="description"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>תיאור</FormLabel>
                                    <FormControl>
                                        <RichTextEditor
                                            value={field.value || ''}
                                            onChange={field.onChange}
                                            placeholder="תיאור מפורט של הפרויקט..."
                                            className="w-full"
                                        />
                                    </FormControl>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />

                        {/* בחירת לקוח */}
                        <FormField
                            control={form.control}
                            name="customer_id"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>לקוח *</FormLabel>
                                    <FormControl>
                                        <CustomerSelect
                                            value={field.value}
                                            onChange={field.onChange}
                                        />
                                    </FormControl>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />

                        {/* בחירת דומיין */}
                        <FormField
                            control={form.control}
                            name="domain_id"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>דומיין (אופציונלי)</FormLabel>
                                    <FormControl>
                                        <DomainSelect
                                            customerId={form.watch('customer_id')}
                                            value={field.value}
                                            onChange={field.onChange}
                                        />
                                    </FormControl>
                                </FormItem>
                            )}
                        />

                        {/* סוג פרויקט */}
                        <FormField
                            control={form.control}
                            name="project_type"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>סוג פרויקט *</FormLabel>
                                    <Select value={field.value} onValueChange={field.onChange}>
                                        <FormControl>
                                            <SelectTrigger>
                                                <SelectValue placeholder="בחר סוג פרויקט" />
                                            </SelectTrigger>
                                        </FormControl>
                                        <SelectContent>
                                            <SelectItem value="website_design">עיצוב אתר</SelectItem>
                                            <SelectItem value="website_development">פיתוח אתר</SelectItem>
                                            <SelectItem value="domain_setup">הגדרת דומיין</SelectItem>
                                            <SelectItem value="email_setup">הגדרת אימייל</SelectItem>
                                            <SelectItem value="maintenance">תחזוקה</SelectItem>
                                            <SelectItem value="hosting_migration">העברת אחסון</SelectItem>
                                            <SelectItem value="seo">קידום SEO</SelectItem>
                                            <SelectItem value="marketing">שיווק</SelectItem>
                                            <SelectItem value="other">אחר</SelectItem>
                                        </SelectContent>
                                    </Select>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />

                        {/* תאריך התחלה */}
                        <FormField
                            control={form.control}
                            name="start_date"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>תאריך התחלה</FormLabel>
                                    <FormControl>
                                        <Input
                                            type="date"
                                            value={formatDateForInput(field.value)}
                                            onChange={(e) => field.onChange(e.target.value)}
                                        />
                                    </FormControl>
                                </FormItem>
                            )}
                        />

                        {/* מועד אחרון */}
                        <FormField
                            control={form.control}
                            name="end_date"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>מועד אחרון</FormLabel>
                                    <FormControl>
                                        <Input
                                            type="date"
                                            value={formatDateForInput(field.value)}
                                            onChange={(e) => field.onChange(e.target.value)}
                                        />
                                    </FormControl>
                                </FormItem>
                            )}
                        />

                        {/* תקציב */}
                        <FormField
                            control={form.control}
                            name="budget"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>תקציב (₪)</FormLabel>
                                    <FormControl>
                                        <div className="relative">
                                            <span className="absolute right-3 top-2.5 text-gray-500">
                                                ₪
                                            </span>
                                            <Input
                                                type="number"
                                                className="pr-8"
                                                placeholder="0"
                                                {...field}
                                                onChange={(e) => field.onChange(parseFloat(e.target.value) || 0)}
                                            />
                                        </div>
                                    </FormControl>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />

                        {/* תגיות */}
                        <FormField
                            control={form.control}
                            name="tags"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>תגיות</FormLabel>
                                    <FormControl>
                                        <Input
                                            placeholder="הקלד תגיות מופרדות בפסיק"
                                            value={(field.value || []).join(', ')}
                                            onChange={(e) => {
                                                const tags = e.target.value.split(',').map(tag => tag.trim()).filter(tag => tag);
                                                field.onChange(tags);
                                            }}
                                        />
                                    </FormControl>
                                </FormItem>
                            )}
                        />

                        {/* סטטוס */}
                        <FormField
                            control={form.control}
                            name="status"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>סטטוס</FormLabel>
                                    <Select value={field.value} onValueChange={field.onChange}>
                                        <FormControl>
                                            <SelectTrigger>
                                                <SelectValue />
                                            </SelectTrigger>
                                        </FormControl>
                                        <SelectContent>
                                            <SelectItem value="open">🟢 פתוח</SelectItem>
                                            <SelectItem value="in_progress">🔵 בביצוע</SelectItem>
                                            <SelectItem value="pending">🟡 ממתין</SelectItem>
                                            <SelectItem value="completed">✅ הושלם</SelectItem>
                                            <SelectItem value="cancelled">❌ בוטל</SelectItem>
                                        </SelectContent>
                                    </Select>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />

                        {/* עדיפות */}
                        <FormField
                            control={form.control}
                            name="priority"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>עדיפות</FormLabel>
                                    <Select value={field.value} onValueChange={field.onChange}>
                                        <FormControl>
                                            <SelectTrigger>
                                                <SelectValue />
                                            </SelectTrigger>
                                        </FormControl>
                                        <SelectContent>
                                            <SelectItem value="low">🟢 נמוכה</SelectItem>
                                            <SelectItem value="medium">🟡 בינונית</SelectItem>
                                            <SelectItem value="high">🟠 גבוהה</SelectItem>
                                            <SelectItem value="urgent">🔴 דחופה</SelectItem>
                                        </SelectContent>
                                    </Select>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />

                        {/* כפתורים */}
                        <div className="flex justify-end gap-3 pt-4">
                            <Button
                                type="button"
                                variant="outline"
                                onClick={() => onOpenChange(false)}
                                disabled={loading}
                            >
                                סגור
                            </Button>
                            {!isViewMode && (
                                <Button
                                    type="submit"
                                    className="bg-blue-500 hover:bg-blue-600"
                                    disabled={loading}
                                >
                                    {loading ? 'שומר...' : '✓ שמור'}
                                </Button>
                            )}
                        </div>
                    </form>
                </Form>
            </DialogContent>
        </Dialog>
    );
}
