'use client';

import { useState, useEffect } from 'react';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';

interface Domain {
    id: string;
    domain: string;
    customer_id: string;
}

interface DomainSelectProps {
    customerId?: string;
    value?: string;
    onChange: (value: string) => void;
    placeholder?: string;
}

export function DomainSelect({ customerId, value, onChange, placeholder = "בחר דומיין" }: DomainSelectProps) {
    const [domains, setDomains] = useState<Domain[]>([]);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        const loadDomains = async () => {
            if (!customerId) {
                setDomains([]);
                setLoading(false);
                return;
            }

            try {
                setLoading(true);
                const response = await fetch(`/api/customers/${customerId}/domains`);
                if (response.ok) {
                    const data = await response.json();
                    setDomains(data.domains || []);
                }
            } catch (error) {
                console.error('Error loading domains:', error);
            } finally {
                setLoading(false);
            }
        };

        loadDomains();
    }, [customerId]);

    if (!customerId) {
        return (
            <Select disabled>
                <SelectTrigger>
                    <SelectValue placeholder="בחר קודם לקוח" />
                </SelectTrigger>
            </Select>
        );
    }

    if (loading) {
        return (
            <Select disabled>
                <SelectTrigger>
                    <SelectValue placeholder="טוען דומיינים..." />
                </SelectTrigger>
            </Select>
        );
    }

    return (
        <Select value={value} onValueChange={onChange}>
            <SelectTrigger>
                <SelectValue placeholder={placeholder} />
            </SelectTrigger>
            <SelectContent>
                {domains.length === 0 ? (
                    <SelectItem value="no-domains" disabled>
                        אין דומיינים ללקוח זה
                    </SelectItem>
                ) : (
                    domains.map((domain) => (
                        <SelectItem key={domain.id} value={domain.id}>
                            {domain.domain}
                        </SelectItem>
                    ))
                )}
            </SelectContent>
        </Select>
    );
}
