'use client';

import { useState, useEffect } from 'react';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';

interface Customer {
    id: string;
    name: string;
    company?: string;
    email?: string;
}

interface CustomerSelectProps {
    value?: string;
    onChange: (value: string) => void;
    placeholder?: string;
}

export function CustomerSelect({ value, onChange, placeholder = "בחר לקוח" }: CustomerSelectProps) {
    const [customers, setCustomers] = useState<Customer[]>([]);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        const loadCustomers = async () => {
            try {
                const response = await fetch('/api/customers');
                if (response.ok) {
                    const data = await response.json();
                    setCustomers(data.customers || []);
                }
            } catch (error) {
                console.error('Error loading customers:', error);
            } finally {
                setLoading(false);
            }
        };

        loadCustomers();
    }, []);

    if (loading) {
        return (
            <Select disabled>
                <SelectTrigger>
                    <SelectValue placeholder="טוען לקוחות..." />
                </SelectTrigger>
            </Select>
        );
    }

    return (
        <Select value={value} onValueChange={onChange}>
            <SelectTrigger>
                <SelectValue placeholder={placeholder} />
            </SelectTrigger>
            <SelectContent>
                {customers.map((customer) => (
                    <SelectItem key={customer.id} value={customer.id}>
                        <div className="flex flex-col">
                            <span className="font-medium">{customer.name}</span>
                            {customer.company && (
                                <span className="text-sm text-gray-500">{customer.company}</span>
                            )}
                        </div>
                    </SelectItem>
                ))}
            </SelectContent>
        </Select>
    );
}
